<?php
/**
 * ============================================================
 * SCMS ADVANCED HEALTH CHECK & DIAGNOSTICS
 * FILE: health.php
 * 
 * Features:
 * - Full directory scan with permissions check
 * - File accessibility testing
 * - Cache clearing (OPcache, APCu, browser)
 * - PHP syntax validation
 * - Module loading test
 * ============================================================
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Handle cache clearing
$cache_cleared = false;
if (isset($_GET['clear_cache'])) {
    // Clear OPcache
    if (function_exists('opcache_reset')) {
        opcache_reset();
        $cache_cleared = true;
    }
    
    // Clear APCu cache
    if (function_exists('apcu_clear_cache')) {
        apcu_clear_cache();
        $cache_cleared = true;
    }
    
    // Clear file stat cache
    clearstatcache(true);
    
    // Force browser cache clear with headers
    header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
    header("Cache-Control: post-check=0, pre-check=0", false);
    header("Pragma: no-cache");
    header("Expires: Sat, 26 Jul 1997 05:00:00 GMT");
}

// Start output buffering for clean display
ob_start();

$health_checks = [];
$critical_issues = 0;
$warnings = 0;
$passed = 0;

/**
 * Add a health check result
 */
function add_check($name, $status, $message, $details = '') {
    global $health_checks, $critical_issues, $warnings, $passed;
    
    $health_checks[] = [
        'name' => $name,
        'status' => $status,
        'message' => $message,
        'details' => $details
    ];
    
    if ($status === 'fail') $critical_issues++;
    elseif ($status === 'warning') $warnings++;
    else $passed++;
}

/**
 * Recursively scan directory and check files
 */
function scan_directory($dir, $base_path = '', &$file_list = []) {
    if (!is_dir($dir)) {
        return $file_list;
    }
    
    $items = scandir($dir);
    
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        
        $full_path = $dir . '/' . $item;
        $relative_path = $base_path . '/' . $item;
        
        if (is_dir($full_path)) {
            $file_list[] = [
                'path' => $relative_path,
                'type' => 'directory',
                'full_path' => $full_path,
                'permissions' => substr(sprintf('%o', fileperms($full_path)), -3),
                'readable' => is_readable($full_path),
                'writable' => is_writable($full_path),
                'size' => 0
            ];
            
            // Recursively scan subdirectory
            scan_directory($full_path, $relative_path, $file_list);
        } else {
            $file_list[] = [
                'path' => $relative_path,
                'type' => 'file',
                'full_path' => $full_path,
                'permissions' => substr(sprintf('%o', fileperms($full_path)), -3),
                'readable' => is_readable($full_path),
                'writable' => is_writable($full_path),
                'size' => filesize($full_path),
                'extension' => pathinfo($full_path, PATHINFO_EXTENSION)
            ];
        }
    }
    
    return $file_list;
}

/**
 * Check PHP syntax of a file
 */
function check_php_syntax($file) {
    $output = [];
    $return_var = 0;
    exec("php -l " . escapeshellarg($file) . " 2>&1", $output, $return_var);
    return $return_var === 0;
}

/**
 * Test if a PHP file can be included
 */
function test_file_include($file) {
    try {
        ob_start();
        $result = @include_once($file);
        ob_end_clean();
        return true;
    } catch (Exception $e) {
        ob_end_clean();
        return false;
    } catch (Error $e) {
        ob_end_clean();
        return false;
    }
}

// ============================================================
// RUN HEALTH CHECKS
// ============================================================

// Check 1: PHP Version
$php_version = phpversion();
if (version_compare($php_version, '7.4', '>=')) {
    add_check('PHP Version', 'pass', "PHP {$php_version} (Good)");
} else {
    add_check('PHP Version', 'fail', "PHP {$php_version} is outdated", "Minimum PHP 7.4 required");
}

// Check 2: Required PHP Extensions
$required_extensions = ['pdo', 'pdo_mysql', 'mbstring', 'json', 'session'];
$missing_extensions = [];
foreach ($required_extensions as $ext) {
    if (!extension_loaded($ext)) {
        $missing_extensions[] = $ext;
    }
}
if (empty($missing_extensions)) {
    add_check('PHP Extensions', 'pass', 'All required extensions loaded');
} else {
    add_check('PHP Extensions', 'fail', 'Missing extensions: ' . implode(', ', $missing_extensions));
}

// Check 3: Database Connection
$db_status = 'Unknown';
$db_details = '';
if (file_exists(__DIR__ . '/config/database.php')) {
    try {
        require_once __DIR__ . '/config/database.php';
        if (isset($pdo) && $pdo instanceof PDO) {
            $stmt = $pdo->query("SELECT VERSION()");
            $mysql_version = $stmt->fetchColumn();
            add_check('Database Connection', 'pass', "Connected to MySQL {$mysql_version}");
            
            // Check tables
            $stmt = $pdo->query("SHOW TABLES");
            $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
            $required_tables = ['users', 'business_profiles', 'sponsored_workers', 'attendance_records'];
            $missing_tables = array_diff($required_tables, $tables);
            
            if (empty($missing_tables)) {
                add_check('Database Tables', 'pass', count($tables) . ' tables found');
            } else {
                add_check('Database Tables', 'warning', 'Missing tables: ' . implode(', ', $missing_tables));
            }
        } else {
            add_check('Database Connection', 'fail', 'PDO object not found in config');
        }
    } catch (Exception $e) {
        add_check('Database Connection', 'fail', 'Connection failed', $e->getMessage());
    }
} else {
    add_check('Database Connection', 'fail', 'config/database.php not found');
}

// Check 4: File System Scan
$base_dir = __DIR__;
$file_list = scan_directory($base_dir, '');

$total_files = 0;
$total_dirs = 0;
$unreadable_files = [];
$wrong_permissions = [];
$php_files = [];

foreach ($file_list as $item) {
    if ($item['type'] === 'directory') {
        $total_dirs++;
        // Directories should be 755
        if ($item['permissions'] !== '755' && !$item['writable']) {
            $wrong_permissions[] = $item['path'] . " ({$item['permissions']})";
        }
    } else {
        $total_files++;
        
        // Check if file is readable
        if (!$item['readable']) {
            $unreadable_files[] = $item['path'];
        }
        
        // Files should be 644 or 755
        if (!in_array($item['permissions'], ['644', '755', '666', '777'])) {
            $wrong_permissions[] = $item['path'] . " ({$item['permissions']})";
        }
        
        // Collect PHP files for syntax check
        if ($item['extension'] === 'php') {
            $php_files[] = $item;
        }
    }
}

if (empty($unreadable_files)) {
    add_check('File Accessibility', 'pass', "{$total_files} files, {$total_dirs} directories - All accessible");
} else {
    add_check('File Accessibility', 'fail', count($unreadable_files) . ' unreadable files', implode(', ', $unreadable_files));
}

if (empty($wrong_permissions)) {
    add_check('File Permissions', 'pass', 'All permissions look good');
} else {
    add_check('File Permissions', 'warning', count($wrong_permissions) . ' files with unusual permissions', implode('<br>', array_slice($wrong_permissions, 0, 10)));
}

// Check 5: Critical Directories
$critical_dirs = [
    'config' => true,  // must exist
    'includes' => true,
    'modules' => true,
    'modules/workers' => false,
    'modules/attendance' => false,
    'modules/admin' => false,
    'uploads' => true,
    'uploads/documents' => false
];

$missing_dirs = [];
$unwritable_dirs = [];

foreach ($critical_dirs as $dir => $required) {
    $dir_path = __DIR__ . '/' . $dir;
    if (!is_dir($dir_path)) {
        if ($required) {
            $missing_dirs[] = $dir;
        }
    } elseif ($dir === 'uploads' || $dir === 'uploads/documents') {
        if (!is_writable($dir_path)) {
            $unwritable_dirs[] = $dir;
        }
    }
}

if (empty($missing_dirs) && empty($unwritable_dirs)) {
    add_check('Directory Structure', 'pass', 'All critical directories present and accessible');
} else {
    if (!empty($missing_dirs)) {
        add_check('Missing Directories', 'fail', 'Missing: ' . implode(', ', $missing_dirs));
    }
    if (!empty($unwritable_dirs)) {
        add_check('Upload Directories', 'fail', 'Not writable: ' . implode(', ', $unwritable_dirs), 'Run: chmod 777 ' . implode(' ', $unwritable_dirs));
    }
}

// Check 6: Critical Files
$critical_files = [
    'config/database.php',
    'includes/functions.php',
    'includes/header.php',
    'includes/footer.php',
    'index.php',
    'dashboard.php',
    'logout.php'
];

$missing_files = [];
foreach ($critical_files as $file) {
    if (!file_exists(__DIR__ . '/' . $file)) {
        $missing_files[] = $file;
    }
}

if (empty($missing_files)) {
    add_check('Core Files', 'pass', 'All core files present');
} else {
    add_check('Core Files', 'fail', 'Missing files: ' . implode(', ', $missing_files));
}

// Check 7: PHP Syntax Validation (sample check)
$syntax_errors = [];
$sample_php_files = array_slice($php_files, 0, 20); // Check first 20 PHP files

foreach ($sample_php_files as $file) {
    if (!check_php_syntax($file['full_path'])) {
        $syntax_errors[] = $file['path'];
    }
}

if (empty($syntax_errors)) {
    add_check('PHP Syntax', 'pass', 'No syntax errors detected (sampled ' . count($sample_php_files) . ' files)');
} else {
    add_check('PHP Syntax', 'fail', 'Syntax errors in: ' . implode(', ', $syntax_errors));
}

// Check 8: Module Files Loading Test
$module_tests = [
    'Workers List' => 'modules/workers/list.php',
    'Workers Add' => 'modules/workers/add.php',
    'Attendance Dashboard' => 'modules/attendance/dashboard.php',
    'System Settings' => 'modules/admin/system_settings.php'
];

$failed_modules = [];
foreach ($module_tests as $name => $path) {
    $full_path = __DIR__ . '/' . $path;
    if (file_exists($full_path)) {
        if (!is_readable($full_path)) {
            $failed_modules[] = "$name (not readable)";
        }
    } else {
        $failed_modules[] = "$name (file missing)";
    }
}

if (empty($failed_modules)) {
    add_check('Module Files', 'pass', 'All module files accessible');
} else {
    add_check('Module Files', 'warning', 'Issues: ' . implode(', ', $failed_modules));
}

// Check 9: Session Functionality
session_start();
$_SESSION['health_test'] = 'working';
if (isset($_SESSION['health_test']) && $_SESSION['health_test'] === 'working') {
    add_check('PHP Sessions', 'pass', 'Sessions working correctly');
    unset($_SESSION['health_test']);
} else {
    add_check('PHP Sessions', 'fail', 'Session test failed');
}

// Check 10: Write Test
$test_file = __DIR__ . '/uploads/health_test_' . time() . '.txt';
$write_test = @file_put_contents($test_file, 'test');
if ($write_test !== false) {
    add_check('File Write Test', 'pass', 'System can write files');
    @unlink($test_file);
} else {
    add_check('File Write Test', 'fail', 'Cannot write to uploads directory', 'Check permissions: chmod 777 uploads');
}

// Check 11: OPcache Status
if (function_exists('opcache_get_status')) {
    $opcache_status = opcache_get_status();
    if ($opcache_status !== false) {
        $cache_hits = $opcache_status['opcache_statistics']['hits'] ?? 0;
        $cache_misses = $opcache_status['opcache_statistics']['misses'] ?? 0;
        $hit_rate = $cache_hits > 0 ? round(($cache_hits / ($cache_hits + $cache_misses)) * 100, 2) : 0;
        add_check('OPcache', 'pass', "Enabled (Hit rate: {$hit_rate}%)", "Cached scripts: " . ($opcache_status['opcache_statistics']['num_cached_scripts'] ?? 0));
    } else {
        add_check('OPcache', 'warning', 'OPcache disabled', 'Enable for better performance');
    }
} else {
    add_check('OPcache', 'warning', 'OPcache not available');
}

// Check 12: Memory Limit
$memory_limit = ini_get('memory_limit');
$memory_int = preg_replace('/[^0-9]/', '', $memory_limit);
if ($memory_int >= 128) {
    add_check('Memory Limit', 'pass', $memory_limit);
} else {
    add_check('Memory Limit', 'warning', $memory_limit . ' (Recommended: 128M or higher)');
}

// Check 13: Max Upload Size
$upload_max = ini_get('upload_max_filesize');
$post_max = ini_get('post_max_size');
add_check('Upload Limits', 'pass', "Max file: {$upload_max}, Max post: {$post_max}");

// Calculate health score
$total_checks = count($health_checks);
$health_percentage = ($total_checks > 0) ? (($passed / $total_checks) * 100) : 0;
$overall_status = $critical_issues > 0 ? 'critical' : ($warnings > 0 ? 'warning' : 'healthy');

// Get buffer content
ob_end_clean();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SCMS Health Check</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            background: #f8f9fa;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            padding: 20px;
        }
        .health-container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .status-card {
            background: white;
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .status-card h2 {
            font-size: 1.5rem;
            margin-bottom: 15px;
            color: #333;
        }
        .check-item {
            display: flex;
            align-items: flex-start;
            padding: 15px;
            margin: 10px 0;
            border-radius: 8px;
            background: #f8f9fa;
            border-left: 4px solid #dee2e6;
        }
        .check-item.pass {
            background: #d4edda;
            border-left-color: #28a745;
        }
        .check-item.warning {
            background: #fff3cd;
            border-left-color: #ffc107;
        }
        .check-item.fail {
            background: #f8d7da;
            border-left-color: #dc3545;
        }
        .check-icon {
            font-size: 24px;
            margin-right: 15px;
            min-width: 24px;
        }
        .check-item.pass .check-icon { color: #28a745; }
        .check-item.warning .check-icon { color: #ffc107; }
        .check-item.fail .check-icon { color: #dc3545; }
        .check-content {
            flex: 1;
        }
        .check-name {
            font-weight: 600;
            margin-bottom: 5px;
            color: #333;
        }
        .check-message {
            color: #666;
            margin-bottom: 5px;
        }
        .check-details {
            font-size: 0.9rem;
            color: #888;
            font-family: 'Courier New', monospace;
            background: rgba(0,0,0,0.05);
            padding: 8px;
            border-radius: 4px;
            margin-top: 8px;
            word-break: break-all;
        }
        .stats {
            display: flex;
            gap: 20px;
            margin: 20px 0;
            flex-wrap: wrap;
        }
        .stat {
            flex: 1;
            min-width: 150px;
            text-align: center;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        .stat-number {
            font-size: 2.5rem;
            font-weight: bold;
            margin-bottom: 5px;
        }
        .stat-label {
            color: #666;
            font-size: 0.9rem;
        }
        .health-score {
            font-size: 3rem;
            font-weight: bold;
            text-align: center;
            margin: 20px 0;
        }
        .health-score.healthy { color: #28a745; }
        .health-score.warning { color: #ffc107; }
        .health-score.critical { color: #dc3545; }
        .btn {
            padding: 10px 20px;
            border-radius: 6px;
            text-decoration: none;
            display: inline-block;
            margin: 5px;
            border: none;
            cursor: pointer;
            font-weight: 500;
        }
        .btn-primary {
            background: #0d6efd;
            color: white;
        }
        .btn-success {
            background: #28a745;
            color: white;
        }
        .btn-danger {
            background: #dc3545;
            color: white;
        }
        .btn-warning {
            background: #ffc107;
            color: #000;
        }
        .alert-success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        .file-scan-summary {
            background: #e7f3ff;
            border: 1px solid #b3d9ff;
            padding: 15px;
            border-radius: 6px;
            margin: 15px 0;
        }
        .progress {
            height: 30px;
            border-radius: 6px;
            overflow: hidden;
            background: #e9ecef;
        }
        .progress-bar {
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            transition: width 0.3s ease;
        }
    </style>
</head>
<body>
    <div class="health-container">
        <!-- Header -->
        <div class="status-card">
            <div style="text-align: center;">
                <h1 style="color: #333; margin-bottom: 10px;">
                    <i class="fas fa-heartbeat"></i> SCMS Health Check
                </h1>
                <p style="color: #666;">System Diagnostics & File Scanner</p>
            </div>
        </div>

        <?php if ($cache_cleared): ?>
        <div class="alert-success">
            <i class="fas fa-check-circle"></i> <strong>Cache Cleared Successfully!</strong>
            OPcache, file cache, and browser cache have been reset.
        </div>
        <?php endif; ?>

        <!-- Overall Status -->
        <div class="status-card">
            <div class="health-score <?= $overall_status ?>">
                <?= round($health_percentage) ?>%
            </div>
            <h2 style="text-align: center;">
                <?php if ($overall_status === 'healthy'): ?>
                    <i class="fas fa-check-circle" style="color: #28a745;"></i> System is Healthy
                <?php elseif ($overall_status === 'warning'): ?>
                    <i class="fas fa-exclamation-triangle" style="color: #ffc107;"></i> System Has Warnings
                <?php else: ?>
                    <i class="fas fa-times-circle" style="color: #dc3545;"></i> System Has Critical Issues
                <?php endif; ?>
            </h2>
            
            <div class="progress" style="margin: 20px 0;">
                <div class="progress-bar <?= $overall_status === 'healthy' ? 'bg-success' : ($overall_status === 'warning' ? 'bg-warning' : 'bg-danger') ?>" 
                     style="width: <?= $health_percentage ?>%">
                    <?= $passed ?> / <?= $total_checks ?> Checks Passed
                </div>
            </div>
            
            <div class="stats">
                <div class="stat">
                    <div class="stat-number" style="color: #28a745;"><?= $passed ?></div>
                    <div class="stat-label">Passed</div>
                </div>
                <div class="stat">
                    <div class="stat-number" style="color: #ffc107;"><?= $warnings ?></div>
                    <div class="stat-label">Warnings</div>
                </div>
                <div class="stat">
                    <div class="stat-number" style="color: #dc3545;"><?= $critical_issues ?></div>
                    <div class="stat-label">Failed</div>
                </div>
            </div>

            <!-- File Scan Summary -->
            <div class="file-scan-summary">
                <h5><i class="fas fa-folder-open"></i> File System Scan Results</h5>
                <div class="row">
                    <div class="col-md-4">
                        <strong>Total Files:</strong> <?= $total_files ?>
                    </div>
                    <div class="col-md-4">
                        <strong>Total Directories:</strong> <?= $total_dirs ?>
                    </div>
                    <div class="col-md-4">
                        <strong>PHP Files:</strong> <?= count($php_files) ?>
                    </div>
                </div>
            </div>
            
            <div style="text-align: center; margin-top: 20px;">
                <a href="?clear_cache=1" class="btn btn-warning" onclick="return confirm('Clear all caches and reload?')">
                    <i class="fas fa-broom"></i> Clear All Caches
                </a>
                <a href="?" class="btn btn-primary">
                    <i class="fas fa-sync"></i> Refresh Check
                </a>
                <?php if ($overall_status === 'healthy'): ?>
                <a href="index.php" class="btn btn-success">
                    <i class="fas fa-sign-in-alt"></i> Go to Login
                </a>
                <?php endif; ?>
            </div>
        </div>

        <!-- Detailed Checks -->
        <div class="status-card">
            <h2><i class="fas fa-list-check"></i> Detailed Health Checks</h2>
            <?php foreach ($health_checks as $check): ?>
                <div class="check-item <?= $check['status'] ?>">
                    <div class="check-icon">
                        <?php if ($check['status'] === 'pass'): ?>
                            <i class="fas fa-check-circle"></i>
                        <?php elseif ($check['status'] === 'warning'): ?>
                            <i class="fas fa-exclamation-triangle"></i>
                        <?php else: ?>
                            <i class="fas fa-times-circle"></i>
                        <?php endif; ?>
                    </div>
                    <div class="check-content">
                        <div class="check-name"><?= htmlspecialchars($check['name']) ?></div>
                        <div class="check-message"><?= htmlspecialchars($check['message']) ?></div>
                        <?php if (!empty($check['details'])): ?>
                            <div class="check-details"><?= nl2br(htmlspecialchars($check['details'])) ?></div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Fix Instructions -->
        <?php if ($critical_issues > 0 || $warnings > 0): ?>
        <div class="status-card">
            <h2><i class="fas fa-tools"></i> Common Fixes</h2>
            
            <?php if (!empty($unreadable_files) || !empty($wrong_permissions)): ?>
            <div class="check-item warning">
                <div class="check-icon"><i class="fas fa-terminal"></i></div>
                <div class="check-content">
                    <div class="check-name">Fix File Permissions</div>
                    <div class="check-message">Run these commands via SSH:</div>
                    <div class="check-details">
cd <?= __DIR__ ?><br>
find . -type f -name "*.php" -exec chmod 644 {} \;<br>
find . -type d -exec chmod 755 {} \;<br>
chmod 777 uploads/<br>
chmod 777 uploads/documents/
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <?php if (!empty($missing_dirs)): ?>
            <div class="check-item fail">
                <div class="check-icon"><i class="fas fa-terminal"></i></div>
                <div class="check-content">
                    <div class="check-name">Create Missing Directories</div>
                    <div class="check-details">
<?php foreach ($missing_dirs as $dir): ?>
mkdir -p <?= $dir ?><br>
<?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <div class="check-item warning">
                <div class="check-icon"><i class="fas fa-info-circle"></i></div>
                <div class="check-content">
                    <div class="check-name">Clear Browser Cache</div>
                    <div class="check-message">Press Ctrl+Shift+R (Windows) or Cmd+Shift+R (Mac) to hard reload</div>
                </div>
            </div>

            <div class="check-item warning">
                <div class="check-icon"><i class="fas fa-info-circle"></i></div>
                <div class="check-content">
                    <div class="check-name">Restart Web Server</div>
                    <div class="check-message">If files still not loading:</div>
                    <div class="check-details">
# Apache<br>
sudo service apache2 restart<br>
<br>
# Nginx + PHP-FPM<br>
sudo service nginx restart<br>
sudo service php-fpm restart
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- System Information -->
        <div class="status-card">
            <h2><i class="fas fa-info-circle"></i> System Information</h2>
            <table class="table table-sm">
                <tr>
                    <td><strong>PHP Version:</strong></td>
                    <td><?= phpversion() ?></td>
                </tr>
                <tr>
                    <td><strong>Server Software:</strong></td>
                    <td><?= $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown' ?></td>
                </tr>
                <tr>
                    <td><strong>Document Root:</strong></td>
                    <td><?= $_SERVER['DOCUMENT_ROOT'] ?? 'Unknown' ?></td>
                </tr>
                <tr>
                    <td><strong>Script Path:</strong></td>
                    <td><?= __DIR__ ?></td>
                </tr>
                <tr>
                    <td><strong>Memory Limit:</strong></td>
                    <td><?= ini_get('memory_limit') ?></td>
                </tr>
                <tr>
                    <td><strong>Max Execution Time:</strong></td>
                    <td><?= ini_get('max_execution_time') ?> seconds</td>
                </tr>
                <tr>
                    <td><strong>Upload Max Filesize:</strong></td>
                    <td><?= ini_get('upload_max_filesize') ?></td>
                </tr>
                <tr>
                    <td><strong>Post Max Size:</strong></td>
                    <td><?= ini_get('post_max_size') ?></td>
                </tr>
            </table>
        </div>

        <!-- Footer -->
        <div style="text-align: center; margin: 30px 0; color: #666;">
            <p>SCMS Health Check v2.0 | Express Remit</p>
            <p style="font-size: 0.9rem;">Last checked: <?= date('Y-m-d H:i:s') ?></p>
        </div>
    </div>

    <script>
        // Auto-refresh every 30 seconds if there are critical issues
        <?php if ($critical_issues > 0): ?>
        setTimeout(() => {
            if (confirm('Critical issues detected. Refresh health check?')) {
                window.location.reload();
            }
        }, 30000);
        <?php endif; ?>
    </script>
</body>
</html>