<?php
/**
 * Admin Dashboard - Modern UI with Dark Mode
 * Main dashboard page showing system overview
 */

session_start();
require_once 'config/database.php';
require_once 'includes/database.php';
require_once 'includes/session_manager.php';
require_once 'includes/system_health.php';
require_once 'includes/resource_manager.php';

// Require authentication
require_authentication();

$page_title = 'Dashboard';
$current_page = 'dashboard';

// Get current user
$current_user = get_logged_in_user();

// Get system statistics
$stats = array(
    'total_workers' => 0,
    'active_workers' => 0,
    'pending_holidays' => 0,
    'expiring_documents' => 0,
    'recent_attendance' => 0
);

try {
    // Total workers
    $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers");
    $stats['total_workers'] = $stmt->fetchColumn();

    // Active workers
    $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active'");
    $stats['active_workers'] = $stmt->fetchColumn();

    // Pending holiday requests
    if (db_table_exists('holiday_requests')) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "holiday_requests WHERE status = 'pending'");
        $stats['pending_holidays'] = $stmt->fetchColumn();
    }

    // Documents expiring in next 30 days
    if (db_table_exists('documents')) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "documents
                             WHERE expiry_date <= DATE_ADD(NOW(), INTERVAL 30 DAY)
                             AND expiry_date >= NOW()");
        $stats['expiring_documents'] = $stmt->fetchColumn();
    }

    // Today's attendance records
    if (db_table_exists('attendance_records')) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "attendance_records WHERE DATE(check_in) = CURDATE()");
        $stats['recent_attendance'] = $stmt->fetchColumn();
    }
} catch (Exception $e) {
    // Silent fail on statistics
}

// Get system capacity
$capacity = get_system_capacity();

include 'includes/header.php';
?>

<div class="dashboard-container">
    <!-- Welcome Section -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-shield-check"></i> Welcome back, <?php echo htmlspecialchars($current_user['first_name'] ?? 'User'); ?>!
            </h1>
            <p class="welcome-subtitle">Here's what's happening with your sponsorship compliance today</p>
        </div>
        <div class="welcome-actions">
            <span class="system-status-badge">
                <i class="bi bi-circle-fill text-success pulse"></i> System Active
            </span>
        </div>
    </div>

    <!-- System Health Alert -->
    <?php if (isset($_SESSION['system_warning'])): ?>
    <div class="alert alert-warning alert-dismissible fade show modern-alert" role="alert">
        <i class="bi bi-exclamation-triangle-fill"></i>
        <strong>System Notice:</strong> <?php echo htmlspecialchars($_SESSION['system_warning']); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Statistics Cards -->
    <div class="stats-grid">
        <!-- Total Workers -->
        <div class="stat-card gradient-blue">
            <div class="stat-icon">
                <i class="fas fa-users"></i>
            </div>
            <div class="stat-details">
                <h3 class="stat-value"><?php echo number_format($stats['total_workers']); ?></h3>
                <p class="stat-label">Total Workers</p>
                <div class="stat-footer">
                    <a href="workers.php" class="stat-link">
                        View all <i class="bi bi-arrow-right"></i>
                    </a>
                </div>
            </div>
            <div class="stat-background">
                <i class="fas fa-users"></i>
            </div>
        </div>

        <!-- Active Workers -->
        <div class="stat-card gradient-green">
            <div class="stat-icon">
                <i class="ri-user-star-fill"></i>
            </div>
            <div class="stat-details">
                <h3 class="stat-value"><?php echo number_format($stats['active_workers']); ?></h3>
                <p class="stat-label">Active Workers</p>
                <div class="stat-footer">
                    <a href="workers.php?status=active" class="stat-link">
                        View active <i class="bi bi-arrow-right"></i>
                    </a>
                </div>
            </div>
            <div class="stat-background">
                <i class="ri-user-star-fill"></i>
            </div>
        </div>

        <!-- Pending Holidays -->
        <div class="stat-card gradient-orange">
            <div class="stat-icon">
                <span class="material-symbols-outlined" style="font-size: 1.75rem;">event_available</span>
            </div>
            <div class="stat-details">
                <h3 class="stat-value"><?php echo number_format($stats['pending_holidays']); ?></h3>
                <p class="stat-label">Pending Holidays</p>
                <div class="stat-footer">
                    <a href="holidays.php?status=pending" class="stat-link">
                        Review requests <i class="bi bi-arrow-right"></i>
                    </a>
                </div>
            </div>
            <div class="stat-background">
                <span class="material-symbols-outlined">event_available</span>
            </div>
        </div>

        <!-- Expiring Documents -->
        <div class="stat-card gradient-red">
            <div class="stat-icon">
                <i class="ri-file-warning-fill"></i>
            </div>
            <div class="stat-details">
                <h3 class="stat-value"><?php echo number_format($stats['expiring_documents']); ?></h3>
                <p class="stat-label">Expiring Soon</p>
                <div class="stat-footer">
                    <a href="documents.php?expiring=30" class="stat-link">
                        View documents <i class="bi bi-arrow-right"></i>
                    </a>
                </div>
            </div>
            <div class="stat-background">
                <i class="ri-file-warning-fill"></i>
            </div>
        </div>
    </div>

    <!-- Attendance -->
    <div class="row g-4 mb-4">
        <div class="col-lg-12">
            <div class="modern-card attendance-card">
                <div class="card-header-modern">
                    <div class="header-content">
                        <h5 class="card-title-modern">
                            <i class="bi bi-clock-history"></i> Today's Attendance
                        </h5>
                        <p class="card-subtitle-modern"><?php echo date('l, F j, Y'); ?></p>
                    </div>
                </div>
                <div class="card-body-modern text-center">
                    <div class="attendance-number">
                        <h2 class="attendance-value"><?php echo number_format($stats['recent_attendance']); ?></h2>
                        <p class="attendance-label">Check-ins recorded</p>
                    </div>
                    <div class="attendance-chart">
                        <div class="mini-chart">
                            <svg viewBox="0 0 100 40" width="100%" height="60">
                                <polyline
                                    fill="none"
                                    stroke="url(#gradient)"
                                    stroke-width="2"
                                    points="0,35 20,30 40,25 60,20 80,15 100,10"
                                />
                                <defs>
                                    <linearGradient id="gradient" x1="0%" y1="0%" x2="100%" y2="0%">
                                        <stop offset="0%" style="stop-color:#667eea;stop-opacity:1" />
                                        <stop offset="100%" style="stop-color:#764ba2;stop-opacity:1" />
                                    </linearGradient>
                                </defs>
                            </svg>
                        </div>
                    </div>
                </div>
                <div class="card-footer-modern">
                    <a href="attendance.php" class="btn-modern-primary w-100">
                        <i class="bi bi-calendar3"></i> View Full Attendance
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="modern-card">
        <div class="card-header-modern">
            <div class="header-content">
                <h5 class="card-title-modern">
                    <i class="bi bi-lightning-charge-fill"></i> Quick Actions
                </h5>
                <p class="card-subtitle-modern">Common tasks and shortcuts</p>
            </div>
        </div>
        <div class="card-body-modern">
            <div class="quick-actions-grid">
                <a href="workers.php?action=add" class="quick-action-btn gradient-cyan">
                    <div class="quick-action-icon">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <span class="quick-action-label">Add Worker</span>
                    <span class="quick-action-arrow"><i class="bi bi-arrow-right"></i></span>
                </a>

                <a href="attendance.php?action=record" class="quick-action-btn gradient-purple">
                    <div class="quick-action-icon">
                        <span class="material-symbols-outlined" style="font-size: 1.5rem;">schedule</span>
                    </div>
                    <span class="quick-action-label">Record Attendance</span>
                    <span class="quick-action-arrow"><i class="bi bi-arrow-right"></i></span>
                </a>

                <a href="documents.php?action=upload" class="quick-action-btn gradient-pink">
                    <div class="quick-action-icon">
                        <i class="ri-upload-cloud-fill"></i>
                    </div>
                    <span class="quick-action-label">Upload Document</span>
                    <span class="quick-action-arrow"><i class="bi bi-arrow-right"></i></span>
                </a>

                <a href="reports.php" class="quick-action-btn gradient-yellow">
                    <div class="quick-action-icon">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <span class="quick-action-label">Generate Report</span>
                    <span class="quick-action-arrow"><i class="bi bi-arrow-right"></i></span>
                </a>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>
