<?php
/**
 * Automatic Update Checker
 * Run via cron: 0 * * * * /usr/bin/php /path/to/cron/update_checker.php
 * Checks for updates pushed from RIU Control Plane and applies them
 */

// Load dependencies
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/riu_helper.php';

echo "[" . date('Y-m-d H:i:s') . "] Starting update checker...\n";

try {
    // Check if already updating
    $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'update_in_progress' LIMIT 1");
    $stmt->execute();
    $updating = $stmt->fetchColumn();

    if ($updating === 'true') {
        echo "[INFO] Update already in progress, skipping\n";
        exit(0);
    }

    // Check for pending operations
    echo "[INFO] Checking for pending operations...\n";
    $operations = check_system_notifications();

    if (!isset($operations['pending']) || !$operations['pending']) {
        echo "[INFO] No pending operations\n";

        // Send heartbeat to show instance is alive
        $worker_count = 0;
        $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active'");
        if ($stmt) {
            $worker_count = $stmt->fetchColumn();
        }

        send_health_ping(array(
            'worker_count' => $worker_count,
            'last_check' => date('Y-m-d H:i:s'),
            'php_version' => PHP_VERSION,
            'database' => 'MySQL ' . $pdo->getAttribute(PDO::ATTR_SERVER_VERSION)
        ));

        exit(0);
    }

    // Process pending operations
    foreach ($operations['operations'] as $operation) {
        $operation_id = $operation['id'];
        $operation_type = $operation['type']; // 'update', 'config_change', 'maintenance'
        $operation_data = $operation['data'];

        echo "[INFO] Processing operation: {$operation_type} (ID: {$operation_id})\n";

        // Mark update as in progress
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                               VALUES ('update_in_progress', 'true')
                               ON DUPLICATE KEY UPDATE setting_value = 'true'");
        $stmt->execute();

        try {
            switch ($operation_type) {
                case 'update':
                    handle_update_operation($operation_id, $operation_data, $pdo);
                    break;

                case 'config_change':
                    handle_config_change($operation_id, $operation_data, $pdo);
                    break;

                case 'maintenance':
                    handle_maintenance_mode($operation_id, $operation_data, $pdo);
                    break;

                default:
                    echo "[WARNING] Unknown operation type: {$operation_type}\n";
                    acknowledge_system_notification($operation_id, 'failed', 'Unknown operation type');
            }

        } catch (Exception $e) {
            echo "[ERROR] Operation failed: " . $e->getMessage() . "\n";
            acknowledge_system_notification($operation_id, 'failed', $e->getMessage());

            // Mark update as complete (failed)
            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "system_settings SET setting_value = 'false' WHERE setting_key = 'update_in_progress'");
            $stmt->execute();
        }
    }

    // Clear update flag
    $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "system_settings SET setting_value = 'false' WHERE setting_key = 'update_in_progress'");
    $stmt->execute();

    echo "[" . date('Y-m-d H:i:s') . "] Update checker completed successfully\n";

} catch (Exception $e) {
    echo "[ERROR] Update checker failed: " . $e->getMessage() . "\n";
    exit(1);
}

exit(0);

// ===== OPERATION HANDLERS =====

function handle_update_operation($operation_id, $data, $pdo) {
    echo "[INFO] Starting system update...\n";

    $target_version = $data['version'] ?? 'unknown';
    $backup_enabled = $data['backup'] ?? true;

    // Report starting update
    log_system_event('update', 'running', "Starting update to version {$target_version}", array(
        'target_version' => $target_version,
        'current_version' => RIU_VERSION
    ));

    // 1. Create backup if enabled
    if ($backup_enabled) {
        echo "[INFO] Creating backup...\n";
        create_backup($pdo);
        log_system_event('update', 'running', 'Backup created', array('progress' => 20));
    }

    // 2. Download update files
    echo "[INFO] Downloading update files...\n";
    $client = init_monitoring_client();
    if (!$client) {
        throw new Exception('Failed to initialize RIU client');
    }

    // Get manifest for target version
    $manifest = $client->getManifest($target_version);
    if (!$manifest || !isset($manifest['files'])) {
        throw new Exception('Failed to get update manifest');
    }

    log_system_event('update', 'running', "Downloading {$manifest['total_files']} files", array('progress' => 30));

    // Download and replace files
    $downloaded = 0;
    foreach ($manifest['files'] as $file) {
        $file_path = __DIR__ . '/../' . $file['path'];
        $file_dir = dirname($file_path);

        // Create directory if needed
        if (!is_dir($file_dir)) {
            mkdir($file_dir, 0755, true);
        }

        // Download file
        $content = $client->downloadFile($target_version, $file['path'], $file_path);
        if ($content === false) {
            throw new Exception("Failed to download: " . $file['path']);
        }

        $downloaded++;
        $progress = 30 + (($downloaded / $manifest['total_files']) * 50);

        if ($downloaded % 10 === 0) {
            log_system_event('update', 'running', "Downloaded {$downloaded}/{$manifest['total_files']} files", array(
                'progress' => round($progress)
            ));
        }
    }

    // 3. Run database migrations if any
    if (isset($data['migrations']) && !empty($data['migrations'])) {
        echo "[INFO] Running database migrations...\n";
        log_system_event('update', 'running', 'Running database migrations', array('progress' => 85));

        run_migrations($data['migrations'], $pdo);
    }

    // 4. Clear cache
    echo "[INFO] Clearing cache...\n";
    clear_application_cache($pdo);

    // 5. Update version number
    $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "system_settings SET setting_value = ? WHERE setting_key = 'system_version'");
    $stmt->execute(array($target_version));

    // Mark update complete
    log_system_event('update', 'completed', "Update to {$target_version} completed successfully", array(
        'progress' => 100,
        'new_version' => $target_version
    ));

    monitor_system_state('active', $target_version, array(
        'updated_at' => date('Y-m-d H:i:s'),
        'updated_from' => RIU_VERSION
    ));

    acknowledge_system_notification($operation_id, 'completed');

    echo "[INFO] Update completed successfully\n";
}

function handle_config_change($operation_id, $data, $pdo) {
    echo "[INFO] Applying configuration changes...\n";

    foreach ($data['settings'] as $key => $value) {
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                               VALUES (?, ?)
                               ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute(array($key, $value, $value));

        echo "[INFO] Updated setting: {$key}\n";
    }

    log_system_event('config', 'completed', 'Configuration updated', array(
        'settings_updated' => count($data['settings'])
    ));

    acknowledge_system_notification($operation_id, 'completed');
}

function handle_maintenance_mode($operation_id, $data, $pdo) {
    $mode = $data['mode'] ?? 'enable'; // enable or disable

    echo "[INFO] Setting maintenance mode: {$mode}\n";

    $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "system_settings SET setting_value = ? WHERE setting_key = 'maintenance_mode'");
    $stmt->execute(array($mode === 'enable' ? 'true' : 'false'));

    $state = $mode === 'enable' ? 'maintenance' : 'active';
    monitor_system_state($state, RIU_VERSION, array(
        'maintenance_mode' => $mode,
        'reason' => $data['reason'] ?? 'Manual maintenance'
    ));

    acknowledge_system_notification($operation_id, 'completed');
}

function create_backup($pdo) {
    $backup_dir = __DIR__ . '/../backups';
    if (!is_dir($backup_dir)) {
        mkdir($backup_dir, 0755, true);
    }

    $timestamp = date('Y-m-d_H-i-s');
    $backup_file = $backup_dir . '/backup_' . $timestamp . '.sql';

    // Simple backup - in production use mysqldump
    exec("mysqldump -u " . DB_USER . " -p" . DB_PASS . " " . DB_NAME . " > {$backup_file}");

    echo "[INFO] Backup created: {$backup_file}\n";
}

function run_migrations($migrations, $pdo) {
    foreach ($migrations as $migration) {
        echo "[INFO] Running migration: {$migration}\n";
        // Execute SQL from migration file or string
        $pdo->exec($migration);
    }
}

function clear_application_cache($pdo) {
    // Clear any cached data
    $cache_dir = __DIR__ . '/../cache';
    if (is_dir($cache_dir)) {
        array_map('unlink', glob("{$cache_dir}/*"));
    }

    // Clear database cache entries
    $pdo->exec("DELETE FROM " . DB_PREFIX . "system_settings WHERE setting_key LIKE 'cache_%'");
}
?>
