<?php
require_once 'config/database.php';

try {
    // Create departments table
    $sql_departments = "CREATE TABLE IF NOT EXISTS " . DB_PREFIX . "departments (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL UNIQUE,
        description TEXT,
        manager_id INT NULL,
        is_active TINYINT(1) DEFAULT 1,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_name (name),
        INDEX idx_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $pdo->exec($sql_departments);
    echo "✓ Departments table created successfully\n";

    // Create job roles table
    $sql_roles = "CREATE TABLE IF NOT EXISTS " . DB_PREFIX . "job_roles (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(100) NOT NULL UNIQUE,
        description TEXT,
        level ENUM('entry', 'junior', 'mid', 'senior', 'manager', 'director', 'executive') DEFAULT 'entry',
        department_id INT NULL,
        is_active TINYINT(1) DEFAULT 1,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (department_id) REFERENCES " . DB_PREFIX . "departments(id) ON DELETE SET NULL,
        INDEX idx_title (title),
        INDEX idx_level (level),
        INDEX idx_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $pdo->exec($sql_roles);
    echo "✓ Job roles table created successfully\n";

    // Insert pre-populated departments
    $departments = [
        ['name' => 'Executive', 'description' => 'Executive leadership team'],
        ['name' => 'Finance', 'description' => 'Financial management and accounting'],
        ['name' => 'Compliance', 'description' => 'Regulatory compliance and legal'],
        ['name' => 'Customer Service', 'description' => 'Customer support and service delivery'],
        ['name' => 'Human Resources', 'description' => 'HR and people management'],
        ['name' => 'Operations', 'description' => 'Day-to-day operational management'],
        ['name' => 'IT & Technology', 'description' => 'Information technology and systems'],
        ['name' => 'Sales & Marketing', 'description' => 'Sales and marketing activities']
    ];

    $stmt_dept = $pdo->prepare("INSERT IGNORE INTO " . DB_PREFIX . "departments (name, description) VALUES (?, ?)");

    foreach ($departments as $dept) {
        $stmt_dept->execute([$dept['name'], $dept['description']]);
    }
    echo "✓ Pre-populated " . count($departments) . " departments\n";

    // Insert pre-populated job roles
    $roles = [
        // Executive level
        ['title' => 'Owner', 'level' => 'executive', 'department' => 'Executive'],
        ['title' => 'Director', 'level' => 'director', 'department' => 'Executive'],
        ['title' => 'Chief Executive Officer (CEO)', 'level' => 'executive', 'department' => 'Executive'],
        ['title' => 'Chief Financial Officer (CFO)', 'level' => 'executive', 'department' => 'Finance'],
        ['title' => 'Chief Operating Officer (COO)', 'level' => 'executive', 'department' => 'Operations'],

        // Management level
        ['title' => 'Manager', 'level' => 'manager', 'department' => null],
        ['title' => 'Finance Manager', 'level' => 'manager', 'department' => 'Finance'],
        ['title' => 'Compliance Manager', 'level' => 'manager', 'department' => 'Compliance'],
        ['title' => 'Customer Service Manager', 'level' => 'manager', 'department' => 'Customer Service'],
        ['title' => 'HR Manager', 'level' => 'manager', 'department' => 'Human Resources'],
        ['title' => 'Operations Manager', 'level' => 'manager', 'department' => 'Operations'],

        // Supervisor level
        ['title' => 'Customer Service Supervisor', 'level' => 'senior', 'department' => 'Customer Service'],
        ['title' => 'Team Supervisor', 'level' => 'senior', 'department' => null],

        // Staff level
        ['title' => 'Customer Assistant', 'level' => 'entry', 'department' => 'Customer Service'],
        ['title' => 'Customer Service Representative', 'level' => 'mid', 'department' => 'Customer Service'],
        ['title' => 'Senior Customer Service Representative', 'level' => 'senior', 'department' => 'Customer Service'],
        ['title' => 'Finance Officer', 'level' => 'mid', 'department' => 'Finance'],
        ['title' => 'Accountant', 'level' => 'mid', 'department' => 'Finance'],
        ['title' => 'Senior Accountant', 'level' => 'senior', 'department' => 'Finance'],
        ['title' => 'Compliance Officer', 'level' => 'mid', 'department' => 'Compliance'],
        ['title' => 'HR Officer', 'level' => 'mid', 'department' => 'Human Resources'],
        ['title' => 'HR Assistant', 'level' => 'junior', 'department' => 'Human Resources'],
        ['title' => 'IT Support Specialist', 'level' => 'mid', 'department' => 'IT & Technology'],
        ['title' => 'Systems Administrator', 'level' => 'senior', 'department' => 'IT & Technology'],
        ['title' => 'Sales Representative', 'level' => 'mid', 'department' => 'Sales & Marketing'],
        ['title' => 'Marketing Coordinator', 'level' => 'mid', 'department' => 'Sales & Marketing']
    ];

    $stmt_role = $pdo->prepare("
        INSERT IGNORE INTO " . DB_PREFIX . "job_roles (title, level, department_id)
        VALUES (?, ?, (SELECT id FROM " . DB_PREFIX . "departments WHERE name = ? LIMIT 1))
    ");

    foreach ($roles as $role) {
        $stmt_role->execute([$role['title'], $role['level'], $role['department']]);
    }
    echo "✓ Pre-populated " . count($roles) . " job roles\n";

    echo "\n✓ All tables created and populated successfully!\n";

} catch (PDOException $e) {
    echo "✗ Error: " . $e->getMessage() . "\n";
    exit(1);
}
?>
