console.log("Address validation script loaded");
class AddressValidator {
    constructor(options = {}) {
        this.postcodeInput = options.postcodeInput;
        this.addressLine1Input = options.addressLine1Input;
        this.addressLine2Input = options.addressLine2Input;
        this.addressLine3Input = options.addressLine3Input;
        this.cityInput = options.cityInput;
        this.countyInput = options.countyInput;
        this.countryInput = options.countryInput;
        this.init();
    }

    init() {
        if (!this.postcodeInput) return;
        this.addLookupButton();
    }

    addLookupButton() {
        const wrapper = document.createElement("div");
        wrapper.className = "input-group";
        const postcodeClone = this.postcodeInput.cloneNode(true);
        const button = document.createElement("button");
        button.type = "button";
        button.className = "btn btn-outline-primary";
        button.innerHTML = '<i class="bi bi-search"></i> Lookup';
        this.postcodeInput.parentNode.replaceChild(wrapper, this.postcodeInput);
        wrapper.appendChild(postcodeClone);
        wrapper.appendChild(button);
        this.postcodeInput = postcodeClone;
        this.triggerButton = button;
        button.addEventListener("click", () => this.lookupPostcode());
    }

    async lookupPostcode() {
        const postcode = this.postcodeInput.value.trim().replace(/\s+/g, "");
        if (!postcode) {
            this.showStatus("Please enter a postcode", "warning");
            return;
        }
        this.setLoading(true);
        try {
            const response = await fetch("https://api.postcodes.io/postcodes/" + postcode);
            const data = await response.json();
            if (data.status === 200 && data.result) {
                this.populateAddress(data.result);
                this.showStatus("Address found!", "success");
            } else {
                this.showStatus("Postcode not found", "danger");
            }
        } catch (error) {
            this.showStatus("Unable to lookup postcode", "danger");
        } finally {
            this.setLoading(false);
        }
    }

    populateAddress(result) {
        if (this.postcodeInput) this.postcodeInput.value = result.postcode;
        if (this.cityInput) this.cityInput.value = result.admin_district || result.post_town || "";
        if (this.countyInput) this.countyInput.value = result.admin_county || "";
        if (this.countryInput) this.countryInput.value = result.country || "England";
        if (this.addressLine1Input && !this.addressLine1Input.value) {
            this.addressLine1Input.placeholder = "Enter house number and street name";
            this.addressLine1Input.focus();
        }
        if (this.addressLine3Input && !this.addressLine3Input.value && result.region) {
            this.addressLine3Input.value = result.region;
        }
        this.highlightFields();
    }

    highlightFields() {
        [this.postcodeInput, this.cityInput, this.countyInput, this.countryInput].filter(f => f).forEach(field => {
            field.classList.add("is-valid");
            setTimeout(() => field.classList.remove("is-valid"), 2000);
        });
    }

    showStatus(message, type = "info") {
        if (!this.statusElement) {
            this.statusElement = document.createElement("div");
            this.statusElement.className = "mt-2";
            this.postcodeInput.parentNode.parentNode.appendChild(this.statusElement);
        }
        const icons = {success: "check-circle", danger: "exclamation-triangle", warning: "exclamation-circle", info: "info-circle"};
        this.statusElement.innerHTML = '<div class="alert alert-' + type + ' alert-dismissible fade show py-2"><i class="bi bi-' + icons[type] + '"></i> ' + message + '<button type="button" class="btn-close py-2" data-bs-dismiss="alert"></button></div>';
        if (type === "success") {
            setTimeout(() => {
                const alert = this.statusElement.querySelector(".alert");
                if (alert) new bootstrap.Alert(alert).close();
            }, 3000);
        }
    }

    setLoading(isLoading) {
        if (this.triggerButton) {
            this.triggerButton.disabled = isLoading;
            this.triggerButton.innerHTML = isLoading ? '<span class="spinner-border spinner-border-sm"></span> Looking up...' : '<i class="bi bi-search"></i> Lookup';
        }
    }
}

document.addEventListener("DOMContentLoaded", function() {
    console.log("DOMContentLoaded fired");
    const postcodeInput = document.querySelector("input[name='postcode']");
    console.log("Postcode input found:", postcodeInput);
    const postcodeInput = document.querySelector("input[name='postcode']");
    if (postcodeInput) {
        window.addressValidator = new AddressValidator({
            postcodeInput: postcodeInput,
            addressLine1Input: document.querySelector("input[name='address_line_1']"),
            addressLine2Input: document.querySelector("input[name='address_line_2']"),
            addressLine3Input: document.querySelector("input[name='address_line_3']"),
            cityInput: document.querySelector("input[name='city']"),
            countyInput: document.querySelector("input[name='county']"),
            countryInput: document.querySelector("input[name='country']")
        });
    }
});
