<?php
session_start();
require_once 'includes/database.php';
require_once 'includes/session_manager.php';

header('Content-Type: application/json');

// Check authentication
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

// Check if user has permission (admin level)
if (($_SESSION['user_role'] ?? 0) < 90) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Insufficient permissions']);
    exit;
}

try {
    $pdo = get_db_connection();

    $action = $_POST['action'] ?? '';
    $module_id = $_POST['module_id'] ?? '';

    if (empty($action) || empty($module_id)) {
        throw new Exception('Missing required parameters');
    }

    // Prevent reordering of dashboard and settings modules
    $locked_modules = ['dashboard', 'settings'];
    if (in_array($module_id, $locked_modules)) {
        throw new Exception('This module cannot be reordered');
    }

    // Get current module's sort_order
    $stmt = $pdo->prepare("SELECT sort_order FROM " . DB_PREFIX . "modules_registry WHERE module_id = ? AND status = 'active'");
    $stmt->execute([$module_id]);
    $current = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$current) {
        throw new Exception('Module not found');
    }

    $current_order = (int)$current['sort_order'];

    if ($action === 'up') {
        // Find the module with the next lower sort_order (move up means lower number)
        // Exclude dashboard from swapping
        $stmt = $pdo->prepare("SELECT module_id, sort_order FROM " . DB_PREFIX . "modules_registry
                               WHERE status = 'active' AND sort_order < ? AND module_id != 'dashboard'
                               ORDER BY sort_order DESC LIMIT 1");
        $stmt->execute([$current_order]);
        $swap_with = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($swap_with) {
            // Swap sort orders
            $pdo->beginTransaction();

            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "modules_registry SET sort_order = ? WHERE module_id = ?");
            $stmt->execute([$swap_with['sort_order'], $module_id]);

            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "modules_registry SET sort_order = ? WHERE module_id = ?");
            $stmt->execute([$current_order, $swap_with['module_id']]);

            $pdo->commit();

            echo json_encode(['success' => true, 'message' => 'Module moved up']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Module is already at the top']);
        }

    } else if ($action === 'down') {
        // Find the module with the next higher sort_order (move down means higher number)
        // Exclude settings from swapping
        $stmt = $pdo->prepare("SELECT module_id, sort_order FROM " . DB_PREFIX . "modules_registry
                               WHERE status = 'active' AND sort_order > ? AND module_id != 'settings'
                               ORDER BY sort_order ASC LIMIT 1");
        $stmt->execute([$current_order]);
        $swap_with = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($swap_with) {
            // Swap sort orders
            $pdo->beginTransaction();

            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "modules_registry SET sort_order = ? WHERE module_id = ?");
            $stmt->execute([$swap_with['sort_order'], $module_id]);

            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "modules_registry SET sort_order = ? WHERE module_id = ?");
            $stmt->execute([$current_order, $swap_with['module_id']]);

            $pdo->commit();

            echo json_encode(['success' => true, 'message' => 'Module moved down']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Module is already at the bottom']);
        }

    } else {
        throw new Exception('Invalid action');
    }

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
