<?php
/**
 * Add New Worker
 */

// Include authentication and system checks
require_once __DIR__ . '/../header_include.php';
require_once __DIR__ . '/../../includes/resource_manager.php';

$errors = array();
$success = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Pre-validate operation
    $validation = validate_data_operation('worker', 'create');

    if (!$validation['valid']) {
        $errors[] = $validation['error'];

        if ($validation['requires_upgrade'] ?? false) {
            $errors[] = 'Please upgrade your license to add more workers.';
        }
    } else {
        // Collect form data
        $full_name = trim($_POST['full_name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $phone = trim($_POST['phone'] ?? '');
        $passport_number = trim($_POST['passport_number'] ?? '');
        $nationality = trim($_POST['nationality'] ?? '');
        $visa_type = trim($_POST['visa_type'] ?? '');
        $job_title = trim($_POST['job_title'] ?? '');
        $start_date = trim($_POST['start_date'] ?? '');

        // Validation
        if (empty($full_name)) {
            $errors[] = 'Full name is required';
        }

        if (empty($passport_number)) {
            $errors[] = 'Passport number is required';
        }

        if (empty($job_title)) {
            $errors[] = 'Job title is required';
        }

        if (empty($start_date)) {
            $errors[] = 'Start date is required';
        }

        // If no errors, proceed with adding worker
        if (empty($errors)) {
            try {
                // Generate worker ID
                $worker_id = 'W' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);

                // Insert worker
                $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "sponsored_workers
                                       (worker_id, full_name, email, phone, passport_number, nationality,
                                        visa_type, job_title, start_date, status, created_at)
                                       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'active', NOW())");

                $stmt->execute(array(
                    $worker_id,
                    $full_name,
                    $email,
                    $phone,
                    $passport_number,
                    $nationality,
                    $visa_type,
                    $job_title,
                    $start_date
                ));

                $new_worker_id = $pdo->lastInsertId();

                // Log operation
                log_resource_operation('add_worker', $new_worker_id, $full_name);

                // Audit log
                $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs
                                       (user_id, action, table_name, record_id, new_values, ip_address, created_at)
                                       VALUES (?, 'create', 'sponsored_workers', ?, ?, ?, NOW())");
                $stmt->execute(array(
                    $current_user['id'],
                    $new_worker_id,
                    $full_name,
                    $_SERVER['REMOTE_ADDR'] ?? 'unknown'
                ));

                $success = 'Worker added successfully!';

                // Clear form
                $_POST = array();

            } catch (Exception $e) {
                $errors[] = 'Failed to add worker. Please try again.';
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Worker - SCMS v3.0</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar would go here -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="bi bi-person-plus-fill"></i> Add New Worker</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="index.php" class="btn btn-sm btn-outline-secondary">
                            <i class="bi bi-arrow-left"></i> Back to Workers
                        </a>
                    </div>
                </div>

                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <i class="bi bi-exclamation-triangle-fill"></i>
                        <strong>Error:</strong>
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if (!empty($success)): ?>
                    <div class="alert alert-success">
                        <i class="bi bi-check-circle-fill"></i>
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <div class="row">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <strong>Worker Information</strong>
                            </div>
                            <div class="card-body">
                                <form method="POST" action="">
                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <label for="full_name" class="form-label">Full Name <span class="text-danger">*</span></label>
                                            <input type="text" class="form-control" id="full_name" name="full_name"
                                                   value="<?php echo htmlspecialchars($_POST['full_name'] ?? ''); ?>" required>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="email" class="form-label">Email</label>
                                            <input type="email" class="form-control" id="email" name="email"
                                                   value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                                        </div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <label for="phone" class="form-label">Phone</label>
                                            <input type="text" class="form-control" id="phone" name="phone"
                                                   value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>">
                                        </div>
                                        <div class="col-md-6">
                                            <label for="passport_number" class="form-label">Passport Number <span class="text-danger">*</span></label>
                                            <input type="text" class="form-control" id="passport_number" name="passport_number"
                                                   value="<?php echo htmlspecialchars($_POST['passport_number'] ?? ''); ?>" required>
                                        </div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <label for="nationality" class="form-label">Nationality</label>
                                            <input type="text" class="form-control" id="nationality" name="nationality"
                                                   value="<?php echo htmlspecialchars($_POST['nationality'] ?? ''); ?>">
                                        </div>
                                        <div class="col-md-6">
                                            <label for="visa_type" class="form-label">Visa Type</label>
                                            <select class="form-select" id="visa_type" name="visa_type">
                                                <option value="">Select...</option>
                                                <option value="Skilled Worker">Skilled Worker</option>
                                                <option value="Health and Care Worker">Health and Care Worker</option>
                                                <option value="Temporary Worker">Temporary Worker</option>
                                                <option value="Other">Other</option>
                                            </select>
                                        </div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <label for="job_title" class="form-label">Job Title <span class="text-danger">*</span></label>
                                            <input type="text" class="form-control" id="job_title" name="job_title"
                                                   value="<?php echo htmlspecialchars($_POST['job_title'] ?? ''); ?>" required>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="start_date" class="form-label">Start Date <span class="text-danger">*</span></label>
                                            <input type="date" class="form-control" id="start_date" name="start_date"
                                                   value="<?php echo htmlspecialchars($_POST['start_date'] ?? ''); ?>" required>
                                        </div>
                                    </div>

                                    <div class="d-grid gap-2">
                                        <button type="submit" class="btn btn-primary btn-lg">
                                            <i class="bi bi-save"></i> Add Worker
                                        </button>
                                        <a href="index.php" class="btn btn-outline-secondary">
                                            <i class="bi bi-x-circle"></i> Cancel
                                        </a>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-4">
                        <?php include __DIR__ . '/../widgets/system_status.php'; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
