<?php
/**
 * Module Management Page
 * View, activate, deactivate, configure modules
 */

session_start();
require_once '../config/database.php';
require_once '../includes/database.php';
require_once '../includes/session_manager.php';
require_once '../includes/ModuleManager.php';

// Require authentication
require_authentication();

// Check admin permission
$current_user = get_logged_in_user();
if ($current_user['permission_level'] < 100) {
    die('Access denied. Admin permission required.');
}

$page_title = 'Module Manager';
$current_page = 'modules';

// Initialize Module Manager
$moduleManager = new ModuleManager($pdo);

// Handle actions
$message = '';
$message_type = 'success';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $module_id = $_POST['module_id'] ?? '';

    switch ($action) {
        case 'scan':
            $count = $moduleManager->scanAndRegister();
            $message = "Scanned modules directory. Found {$count} module(s).";
            break;

        case 'activate':
            $result = $moduleManager->activate($module_id);
            $message = $result['message'];
            $message_type = $result['success'] ? 'success' : 'danger';
            break;

        case 'deactivate':
            $result = $moduleManager->deactivate($module_id);
            $message = $result['message'];
            $message_type = $result['success'] ? 'success' : 'danger';
            break;

        case 'uninstall':
            if (isset($_POST['confirm']) && $_POST['confirm'] === 'yes') {
                $result = $moduleManager->uninstall($module_id);
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'danger';
            }
            break;

        case 'update_settings':
            $settings = $_POST['settings'] ?? array();
            $result = $moduleManager->updateSettings($module_id, $settings);
            $message = $result['message'];
            $message_type = $result['success'] ? 'success' : 'danger';
            break;
    }

    // Reload modules after action
    $moduleManager = new ModuleManager($pdo);
}

// Get all modules
$modules = $moduleManager->getAllModules();

// Separate active and inactive
$active_modules = array();
$inactive_modules = array();

foreach ($modules as $module) {
    if ($module['status'] === 'active') {
        $active_modules[] = $module;
    } else {
        $inactive_modules[] = $module;
    }
}

include 'header.php';
?>

<div class="dashboard-container">
    <!-- Page Header -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-puzzle"></i> Module Manager
            </h1>
            <p class="welcome-subtitle">Manage system modules - activate, deactivate, and configure features</p>
        </div>
        <div class="welcome-actions">
            <form method="POST" class="d-inline">
                <input type="hidden" name="action" value="scan">
                <button type="submit" class="btn-modern-primary">
                    <i class="bi bi-arrow-clockwise"></i> Scan for Modules
                </button>
            </form>
        </div>
    </div>

    <!-- Alert Message -->
    <?php if ($message): ?>
    <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show modern-alert" role="alert">
        <i class="bi bi-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>-fill"></i>
        <?php echo htmlspecialchars($message); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Active Modules -->
    <div class="modern-card mb-4">
        <div class="card-header-modern">
            <div class="header-content">
                <h5 class="card-title-modern">
                    <i class="bi bi-check-circle-fill text-success"></i> Active Modules
                </h5>
                <p class="card-subtitle-modern"><?php echo count($active_modules); ?> module(s) currently active</p>
            </div>
        </div>
        <div class="card-body-modern">
            <?php if (empty($active_modules)): ?>
                <div class="text-center py-5">
                    <i class="bi bi-inbox" style="font-size: 3rem; color: var(--text-muted);"></i>
                    <p class="text-muted mt-3">No active modules. Activate modules from the available list below.</p>
                </div>
            <?php else: ?>
                <div class="row g-4">
                    <?php foreach ($active_modules as $module):
                        $info = $moduleManager->getModuleInfo($module['module_id']);
                    ?>
                    <div class="col-md-6 col-lg-4">
                        <div class="module-card active">
                            <div class="module-header">
                                <div class="module-icon gradient-green">
                                    <i class="bi <?php echo htmlspecialchars($module['icon'] ?? 'bi-puzzle'); ?>"></i>
                                </div>
                                <span class="module-badge active">Active</span>
                            </div>
                            <div class="module-body">
                                <h6 class="module-title"><?php echo htmlspecialchars($module['name']); ?></h6>
                                <p class="module-description"><?php echo htmlspecialchars($module['description']); ?></p>
                                <div class="module-meta">
                                    <span class="module-version">
                                        <i class="bi bi-tag"></i> v<?php echo htmlspecialchars($module['version']); ?>
                                    </span>
                                    <span class="module-author">
                                        <i class="bi bi-person"></i> <?php echo htmlspecialchars($module['author'] ?? 'SCMS'); ?>
                                    </span>
                                </div>
                            </div>
                            <div class="module-footer">
                                <div class="btn-group w-100" role="group">
                                    <?php if ($info && isset($info['has_settings']) && $info['has_settings']): ?>
                                    <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#settingsModal<?php echo $module['module_id']; ?>">
                                        <i class="bi bi-gear"></i> Settings
                                    </button>
                                    <?php endif; ?>
                                    <form method="POST" class="flex-fill" onsubmit="return confirm('Deactivate this module?');">
                                        <input type="hidden" name="action" value="deactivate">
                                        <input type="hidden" name="module_id" value="<?php echo $module['module_id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-outline-warning w-100">
                                            <i class="bi bi-pause-circle"></i> Deactivate
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>

                        <!-- Settings Modal -->
                        <?php if ($info && isset($info['has_settings']) && $info['has_settings']): ?>
                        <div class="modal fade" id="settingsModal<?php echo $module['module_id']; ?>" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <div class="modal-header">
                                        <h5 class="modal-title"><?php echo htmlspecialchars($module['name']); ?> Settings</h5>
                                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                    </div>
                                    <form method="POST">
                                        <div class="modal-body">
                                            <input type="hidden" name="action" value="update_settings">
                                            <input type="hidden" name="module_id" value="<?php echo $module['module_id']; ?>">

                                            <?php
                                            $current_settings = $moduleManager->getSettings($module['module_id']);
                                            if (isset($info['settings_fields'])):
                                                foreach ($info['settings_fields'] as $field):
                                            ?>
                                                <div class="mb-3">
                                                    <label class="form-label"><?php echo htmlspecialchars($field['label']); ?></label>
                                                    <?php if ($field['type'] === 'text'): ?>
                                                        <input type="text"
                                                               class="form-control"
                                                               name="settings[<?php echo $field['name']; ?>]"
                                                               value="<?php echo htmlspecialchars($current_settings[$field['name']] ?? $field['default'] ?? ''); ?>">
                                                    <?php elseif ($field['type'] === 'checkbox'): ?>
                                                        <div class="form-check">
                                                            <input type="checkbox"
                                                                   class="form-check-input"
                                                                   name="settings[<?php echo $field['name']; ?>]"
                                                                   value="1"
                                                                   <?php echo (!empty($current_settings[$field['name']]) ? 'checked' : ''); ?>>
                                                            <label class="form-check-label"><?php echo htmlspecialchars($field['help'] ?? ''); ?></label>
                                                        </div>
                                                    <?php elseif ($field['type'] === 'select'): ?>
                                                        <select class="form-select" name="settings[<?php echo $field['name']; ?>]">
                                                            <?php foreach ($field['options'] as $value => $label): ?>
                                                                <option value="<?php echo htmlspecialchars($value); ?>"
                                                                    <?php echo ($current_settings[$field['name']] ?? '') == $value ? 'selected' : ''; ?>>
                                                                    <?php echo htmlspecialchars($label); ?>
                                                                </option>
                                                            <?php endforeach; ?>
                                                        </select>
                                                    <?php endif; ?>
                                                    <?php if (isset($field['help'])): ?>
                                                        <small class="form-text text-muted"><?php echo htmlspecialchars($field['help']); ?></small>
                                                    <?php endif; ?>
                                                </div>
                                            <?php
                                                endforeach;
                                            endif;
                                            ?>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                            <button type="submit" class="btn btn-primary">Save Settings</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Available Modules -->
    <div class="modern-card">
        <div class="card-header-modern">
            <div class="header-content">
                <h5 class="card-title-modern">
                    <i class="bi bi-box-seam"></i> Available Modules
                </h5>
                <p class="card-subtitle-modern"><?php echo count($inactive_modules); ?> module(s) available for activation</p>
            </div>
        </div>
        <div class="card-body-modern">
            <?php if (empty($inactive_modules)): ?>
                <div class="text-center py-5">
                    <i class="bi bi-check-circle" style="font-size: 3rem; color: var(--text-success);"></i>
                    <p class="text-muted mt-3">All available modules are active!</p>
                </div>
            <?php else: ?>
                <div class="row g-4">
                    <?php foreach ($inactive_modules as $module): ?>
                    <div class="col-md-6 col-lg-4">
                        <div class="module-card inactive">
                            <div class="module-header">
                                <div class="module-icon gradient-blue">
                                    <i class="bi <?php echo htmlspecialchars($module['icon'] ?? 'bi-puzzle'); ?>"></i>
                                </div>
                                <span class="module-badge inactive">Inactive</span>
                            </div>
                            <div class="module-body">
                                <h6 class="module-title"><?php echo htmlspecialchars($module['name']); ?></h6>
                                <p class="module-description"><?php echo htmlspecialchars($module['description']); ?></p>
                                <div class="module-meta">
                                    <span class="module-version">
                                        <i class="bi bi-tag"></i> v<?php echo htmlspecialchars($module['version']); ?>
                                    </span>
                                    <span class="module-author">
                                        <i class="bi bi-person"></i> <?php echo htmlspecialchars($module['author'] ?? 'SCMS'); ?>
                                    </span>
                                </div>
                            </div>
                            <div class="module-footer">
                                <div class="btn-group w-100" role="group">
                                    <form method="POST" class="flex-fill">
                                        <input type="hidden" name="action" value="activate">
                                        <input type="hidden" name="module_id" value="<?php echo $module['module_id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-success w-100">
                                            <i class="bi bi-play-circle"></i> Activate
                                        </button>
                                    </form>
                                    <button class="btn btn-sm btn-outline-danger"
                                            data-bs-toggle="modal"
                                            data-bs-target="#uninstallModal<?php echo $module['module_id']; ?>">
                                        <i class="bi bi-trash"></i> Uninstall
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Uninstall Confirmation Modal -->
                        <div class="modal fade" id="uninstallModal<?php echo $module['module_id']; ?>" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <div class="modal-header bg-danger text-white">
                                        <h5 class="modal-title"><i class="bi bi-exclamation-triangle"></i> Confirm Uninstall</h5>
                                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                                    </div>
                                    <form method="POST">
                                        <div class="modal-body">
                                            <input type="hidden" name="action" value="uninstall">
                                            <input type="hidden" name="module_id" value="<?php echo $module['module_id']; ?>">
                                            <input type="hidden" name="confirm" value="yes">

                                            <p class="mb-3">
                                                <strong>Are you sure you want to uninstall "<?php echo htmlspecialchars($module['name']); ?>"?</strong>
                                            </p>
                                            <div class="alert alert-warning">
                                                <i class="bi bi-info-circle"></i> This will remove the module and all its data. This action cannot be undone.
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                            <button type="submit" class="btn btn-danger">Yes, Uninstall</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
.module-card {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 15px;
    overflow: hidden;
    transition: all 0.3s ease;
}

.module-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 12px 24px var(--shadow-lg);
}

.module-card.active {
    border-top: 3px solid #22c55e;
}

.module-card.inactive {
    border-top: 3px solid #94a3b8;
}

.module-header {
    padding: 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
}

.module-icon {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.75rem;
}

.module-badge {
    padding: 0.35rem 0.75rem;
    border-radius: 50px;
    font-size: 0.75rem;
    font-weight: 600;
}

.module-badge.active {
    background: rgba(34, 197, 94, 0.1);
    color: #22c55e;
}

.module-badge.inactive {
    background: rgba(148, 163, 184, 0.1);
    color: #94a3b8;
}

.module-body {
    padding: 0 1.5rem 1.5rem;
}

.module-title {
    font-size: 1.1rem;
    font-weight: 700;
    color: var(--text-primary);
    margin-bottom: 0.5rem;
}

.module-description {
    color: var(--text-secondary);
    font-size: 0.9rem;
    margin-bottom: 1rem;
    line-height: 1.5;
}

.module-meta {
    display: flex;
    gap: 1rem;
    font-size: 0.8rem;
    color: var(--text-muted);
}

.module-footer {
    padding: 1rem 1.5rem;
    border-top: 1px solid var(--border-color);
}
</style>

<?php include 'footer.php'; ?>
