<?php
/**
 * Admin Dashboard
 * Main dashboard page showing system overview
 */

session_start();
require_once '../config/database.php';
require_once '../includes/database.php';
require_once '../includes/session_manager.php';
require_once '../includes/system_health.php';
require_once '../includes/resource_manager.php';

// Require authentication
require_authentication();

$page_title = 'Dashboard';
$current_page = 'dashboard';

// Get current user
$current_user = get_current_user();

// Get system statistics
$stats = array(
    'total_workers' => 0,
    'active_workers' => 0,
    'pending_holidays' => 0,
    'expiring_documents' => 0,
    'recent_attendance' => 0
);

try {
    // Total workers
    $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers");
    $stats['total_workers'] = $stmt->fetchColumn();

    // Active workers
    $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active'");
    $stats['active_workers'] = $stmt->fetchColumn();

    // Pending holiday requests
    if (db_table_exists('holiday_requests')) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "holiday_requests WHERE status = 'pending'");
        $stats['pending_holidays'] = $stmt->fetchColumn();
    }

    // Documents expiring in next 30 days
    if (db_table_exists('documents')) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "documents
                             WHERE expiry_date <= DATE_ADD(NOW(), INTERVAL 30 DAY)
                             AND expiry_date >= NOW()");
        $stats['expiring_documents'] = $stmt->fetchColumn();
    }

    // Today's attendance records
    if (db_table_exists('attendance_records')) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "attendance_records WHERE DATE(check_in) = CURDATE()");
        $stats['recent_attendance'] = $stmt->fetchColumn();
    }
} catch (Exception $e) {
    // Silent fail on statistics
}

// Get system capacity
$capacity = get_system_capacity();

include 'header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-12">
            <h1 class="h3 mb-0">
                <i class="bi bi-speedometer2"></i> Dashboard
            </h1>
            <p class="text-muted mb-0">Welcome back, <?php echo htmlspecialchars($current_user['full_name']); ?></p>
        </div>
    </div>

    <!-- System Health Alert -->
    <?php if (isset($_SESSION['system_warning'])): ?>
    <div class="alert alert-warning alert-dismissible fade show" role="alert">
        <i class="bi bi-exclamation-triangle-fill"></i>
        <strong>System Notice:</strong> <?php echo htmlspecialchars($_SESSION['system_warning']); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Statistics Cards -->
    <div class="row g-4 mb-4">
        <!-- Total Workers -->
        <div class="col-md-3">
            <div class="card border-0 shadow-sm h-100">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <p class="text-muted mb-1 small">Total Workers</p>
                            <h3 class="mb-0"><?php echo number_format($stats['total_workers']); ?></h3>
                        </div>
                        <div class="bg-primary bg-opacity-10 rounded p-3">
                            <i class="bi bi-people-fill text-primary" style="font-size: 2rem;"></i>
                        </div>
                    </div>
                </div>
                <div class="card-footer bg-transparent border-0 pt-0">
                    <a href="workers.php" class="text-decoration-none small">
                        View all <i class="bi bi-arrow-right"></i>
                    </a>
                </div>
            </div>
        </div>

        <!-- Active Workers -->
        <div class="col-md-3">
            <div class="card border-0 shadow-sm h-100">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <p class="text-muted mb-1 small">Active Workers</p>
                            <h3 class="mb-0 text-success"><?php echo number_format($stats['active_workers']); ?></h3>
                        </div>
                        <div class="bg-success bg-opacity-10 rounded p-3">
                            <i class="bi bi-person-check-fill text-success" style="font-size: 2rem;"></i>
                        </div>
                    </div>
                </div>
                <div class="card-footer bg-transparent border-0 pt-0">
                    <a href="workers.php?status=active" class="text-decoration-none small">
                        View active <i class="bi bi-arrow-right"></i>
                    </a>
                </div>
            </div>
        </div>

        <!-- Pending Holidays -->
        <div class="col-md-3">
            <div class="card border-0 shadow-sm h-100">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <p class="text-muted mb-1 small">Pending Holidays</p>
                            <h3 class="mb-0 text-warning"><?php echo number_format($stats['pending_holidays']); ?></h3>
                        </div>
                        <div class="bg-warning bg-opacity-10 rounded p-3">
                            <i class="bi bi-calendar-event-fill text-warning" style="font-size: 2rem;"></i>
                        </div>
                    </div>
                </div>
                <div class="card-footer bg-transparent border-0 pt-0">
                    <a href="holidays.php?status=pending" class="text-decoration-none small">
                        Review requests <i class="bi bi-arrow-right"></i>
                    </a>
                </div>
            </div>
        </div>

        <!-- Expiring Documents -->
        <div class="col-md-3">
            <div class="card border-0 shadow-sm h-100">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <p class="text-muted mb-1 small">Expiring Soon</p>
                            <h3 class="mb-0 text-danger"><?php echo number_format($stats['expiring_documents']); ?></h3>
                        </div>
                        <div class="bg-danger bg-opacity-10 rounded p-3">
                            <i class="bi bi-file-earmark-text-fill text-danger" style="font-size: 2rem;"></i>
                        </div>
                    </div>
                </div>
                <div class="card-footer bg-transparent border-0 pt-0">
                    <a href="documents.php?expiring=30" class="text-decoration-none small">
                        View documents <i class="bi bi-arrow-right"></i>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- License Capacity -->
    <div class="row g-4 mb-4">
        <div class="col-md-8">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-0 py-3">
                    <h5 class="mb-0"><i class="bi bi-graph-up"></i> License Capacity</h5>
                </div>
                <div class="card-body">
                    <div class="d-flex justify-content-between mb-2">
                        <span class="text-muted">Worker Usage</span>
                        <span class="fw-bold">
                            <?php echo $capacity['current_usage']; ?> / <?php echo $capacity['capacity']; ?>
                            (<?php echo $capacity['percentage']; ?>%)
                        </span>
                    </div>
                    <div class="progress" style="height: 25px;">
                        <div class="progress-bar <?php echo $capacity['percentage'] > 80 ? 'bg-danger' : ($capacity['percentage'] > 60 ? 'bg-warning' : 'bg-success'); ?>"
                             role="progressbar"
                             style="width: <?php echo $capacity['percentage']; ?>%">
                            <?php echo $capacity['percentage']; ?>%
                        </div>
                    </div>
                    <p class="text-muted small mt-2 mb-0">
                        License Type: <strong><?php echo ucfirst($capacity['license_type']); ?></strong>
                    </p>
                </div>
            </div>
        </div>

        <div class="col-md-4">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-0 py-3">
                    <h5 class="mb-0"><i class="bi bi-calendar3"></i> Today's Attendance</h5>
                </div>
                <div class="card-body text-center">
                    <h2 class="text-primary mb-0"><?php echo number_format($stats['recent_attendance']); ?></h2>
                    <p class="text-muted small mb-0">Check-ins recorded today</p>
                </div>
                <div class="card-footer bg-transparent border-0">
                    <a href="attendance.php" class="btn btn-outline-primary btn-sm w-100">
                        <i class="bi bi-clock-history"></i> View Attendance
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="row g-4">
        <div class="col-12">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-0 py-3">
                    <h5 class="mb-0"><i class="bi bi-lightning-charge-fill"></i> Quick Actions</h5>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <a href="workers.php?action=add" class="btn btn-outline-primary w-100">
                                <i class="bi bi-person-plus-fill"></i><br>Add Worker
                            </a>
                        </div>
                        <div class="col-md-3">
                            <a href="attendance.php?action=record" class="btn btn-outline-success w-100">
                                <i class="bi bi-clock-fill"></i><br>Record Attendance
                            </a>
                        </div>
                        <div class="col-md-3">
                            <a href="documents.php?action=upload" class="btn btn-outline-info w-100">
                                <i class="bi bi-file-earmark-arrow-up-fill"></i><br>Upload Document
                            </a>
                        </div>
                        <div class="col-md-3">
                            <a href="reports.php" class="btn btn-outline-warning w-100">
                                <i class="bi bi-file-earmark-bar-graph-fill"></i><br>Generate Report
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>
