<?php
$page_title = 'Workers List';
require_once '../../includes/header.php';

// Search and filter
$search = $_GET['search'] ?? '';
$status = $_GET['status'] ?? '';
$department = $_GET['department'] ?? '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 25;

// Build query
$where = ["sw.business_profile_id = ?"];
$params = [$_SESSION['business_profile_id']];

if ($search) {
    $where[] = "(sw.first_name LIKE ? OR sw.last_name LIKE ? OR sw.employee_number LIKE ?)";
    $search_term = "%{$search}%";
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
}

if ($status) {
    $where[] = "sw.employment_status = ?";
    $params[] = $status;
}

if ($department) {
    $where[] = "sw.department = ?";
    $params[] = $department;
}

$where_clause = implode(' AND ', $where);

// Count total
$count_sql = "SELECT COUNT(*) FROM sponsored_workers sw WHERE $where_clause";
$total = $pdo->prepare($count_sql);
$total->execute($params);
$total_records = $total->fetchColumn();

$pagination = paginate($total_records, $per_page, $page);

// Fetch workers
$sql = "SELECT sw.*, CONCAT(sw.first_name, ' ', sw.last_name) as full_name,
        DATEDIFF(sw.visa_expiry_date, CURDATE()) as days_to_expiry
        FROM sponsored_workers sw
        WHERE $where_clause
        ORDER BY sw.last_name, sw.first_name
        LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$workers = $stmt->fetchAll();

// Get filter options
$statuses = get_list_items($pdo, 'employment_status');
$departments = get_list_items($pdo, 'departments');
?>

<div class="stat-card">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5><i class="bi bi-people"></i> Workers (<?= $total_records ?>)</h5>
        <a href="add.php" class="btn btn-primary">
            <i class="bi bi-person-plus"></i> Add New Worker
        </a>
    </div>
    
    <!-- Search and Filter -->
    <form method="GET" class="row g-3 mb-4">
        <div class="col-md-4">
            <input type="text" name="search" class="form-control" 
                   placeholder="Search name or employee number..." 
                   value="<?= htmlspecialchars($search) ?>">
        </div>
        <div class="col-md-3">
            <select name="status" class="form-select">
                <option value="">All Statuses</option>
                <?php foreach ($statuses as $value => $label): ?>
                    <option value="<?= $value ?>" <?= $status === $value ? 'selected' : '' ?>>
                        <?= htmlspecialchars($label) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-3">
            <select name="department" class="form-select">
                <option value="">All Departments</option>
                <?php foreach ($departments as $value => $label): ?>
                    <option value="<?= $value ?>" <?= $department === $value ? 'selected' : '' ?>>
                        <?= htmlspecialchars($label) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-2">
            <button type="submit" class="btn btn-secondary w-100">
                <i class="bi bi-search"></i> Search
            </button>
        </div>
    </form>
    
    <!-- Workers Table -->
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>Employee #</th>
                    <th>Name</th>
                    <th>Job Title</th>
                    <th>Department</th>
                    <th>Visa Status</th>
                    <th>Employment Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($workers)): ?>
                <tr>
                    <td colspan="7" class="text-center py-4 text-muted">
                        <i class="bi bi-inbox" style="font-size: 48px;"></i>
                        <p class="mt-2">No workers found</p>
                    </td>
                </tr>
                <?php else: ?>
                <?php foreach ($workers as $worker): ?>
                <tr>
                    <td><?= htmlspecialchars($worker['employee_number']) ?></td>
                    <td>
                        <strong><?= htmlspecialchars($worker['full_name']) ?></strong>
                    </td>
                    <td><?= htmlspecialchars($worker['job_title']) ?></td>
                    <td><?= htmlspecialchars($worker['department']) ?></td>
                    <td>
                        <?php if ($worker['days_to_expiry'] !== null): ?>
                            <?php if ($worker['days_to_expiry'] < 0): ?>
                                <span class="badge bg-danger">Expired</span>
                            <?php elseif ($worker['days_to_expiry'] <= 30): ?>
                                <span class="badge bg-danger"><?= $worker['days_to_expiry'] ?> days</span>
                            <?php elseif ($worker['days_to_expiry'] <= 90): ?>
                                <span class="badge bg-warning"><?= $worker['days_to_expiry'] ?> days</span>
                            <?php else: ?>
                                <span class="badge bg-success">Valid</span>
                            <?php endif; ?>
                        <?php else: ?>
                            <span class="badge bg-secondary">N/A</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <span class="badge <?= $worker['employment_status'] === 'Active' ? 'bg-success' : 'bg-secondary' ?>">
                            <?= htmlspecialchars($worker['employment_status']) ?>
                        </span>
                    </td>
                    <td>
                        <a href="view.php?id=<?= $worker['id'] ?>" class="btn btn-sm btn-outline-primary" title="View">
                            <i class="bi bi-eye"></i>
                        </a>
                        <a href="edit.php?id=<?= $worker['id'] ?>" class="btn btn-sm btn-outline-warning" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Pagination -->
    <?php if ($pagination['total_pages'] > 1): ?>
    <nav>
        <ul class="pagination justify-content-center">
            <?php if ($pagination['has_prev']): ?>
            <li class="page-item">
                <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    Previous
                </a>
            </li>
            <?php endif; ?>
            
            <?php for ($i = 1; $i <= $pagination['total_pages']; $i++): ?>
            <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    <?= $i ?>
                </a>
            </li>
            <?php endfor; ?>
            
            <?php if ($pagination['has_next']): ?>
            <li class="page-item">
                <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    Next
                </a>
            </li>
            <?php endif; ?>
        </ul>
    </nav>
    <?php endif; ?>
</div>

<?php require_once '../../includes/footer.php'; ?>