
/* ============================================================
   FILE 10: modules/workers/edit.php
   Edit existing worker
   ============================================================ */

<?php
require __DIR__ . '/../../config/database.php';
require __DIR__ . '/../../includes/functions.php';

require_login();
require_permission(70);

$worker_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

$stmt = $pdo->prepare("SELECT * FROM sponsored_workers WHERE id = ? AND deleted_at IS NULL");
$stmt->execute([$worker_id]);
$worker = $stmt->fetch();

if (!$worker) {
    die('Worker not found');
}

$page_title = 'Edit Worker';
$success = '';
$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $stmt = $pdo->prepare("UPDATE sponsored_workers SET 
            first_name = ?, last_name = ?, email = ?, phone = ?,
            visa_type = ?, job_title = ?, department = ?,
            employment_status = ?, visa_expiry_date = ?,
            annual_salary = ?, employment_start_date = ?,
            updated_at = NOW()
            WHERE id = ?");
        
        $stmt->execute([
            $_POST['first_name'] ?? null,
            $_POST['last_name'] ?? null,
            $_POST['email'] ?? null,
            $_POST['phone'] ?? null,
            $_POST['visa_type'] ?? null,
            $_POST['job_title'] ?? null,
            $_POST['department'] ?? null,
            $_POST['employment_status'] ?? 'Active',
            uk_date_to_db($_POST['visa_expiry_date'] ?? null),
            $_POST['annual_salary'] ?? null,
            uk_date_to_db($_POST['employment_start_date'] ?? null),
            $worker_id
        ]);
        
        log_audit($user['id'], 'UPDATE', 'sponsored_workers', $worker_id);
        
        $_SESSION['flash']['success'] = 'Worker updated successfully!';
        header('Location: ' . base_url('modules/workers/view.php?id=' . $worker_id));
        exit;
        
    } catch (Exception $e) {
        $errors[] = $e->getMessage();
    }
}

$visa_types = get_list_items('visa_types');
$employment_statuses = get_list_items('employment_status');
$departments = get_list_items('departments');

include __DIR__ . '/../../includes/header.php';
?>

<div class="py-4">
    <div class="mb-4">
        <h1 class="h3 mb-1">Edit Worker</h1>
        <p class="text-muted mb-0"><?= e($worker['employee_number']) ?></p>
    </div>
    
    <?php if (!empty($errors)): ?>
        <div class="alert alert-danger">
            <ul class="mb-0">
                <?php foreach ($errors as $error): ?>
                    <li><?= e($error) ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>
    
    <form method="POST">
        <div class="card">
            <div class="card-header bg-white border-0 py-3">
                <h5 class="mb-0 fw-bold">Worker Information</h5>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">First Name</label>
                        <input type="text" class="form-control" name="first_name" 
                               value="<?= e($worker['first_name']) ?>">
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Last Name</label>
                        <input type="text" class="form-control" name="last_name" 
                               value="<?= e($worker['last_name']) ?>">
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Email</label>
                        <input type="email" class="form-control" name="email" 
                               value="<?= e($worker['email']) ?>">
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Phone</label>
                        <input type="tel" class="form-control" name="phone" 
                               value="<?= e($worker['phone']) ?>">
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Visa Type</label>
                        <select class="form-select" name="visa_type">
                            <option value="">Select...</option>
                            <?php foreach ($visa_types as $visa): ?>
                                <option value="<?= e($visa['item_value']) ?>" 
                                        <?= $worker['visa_type'] === $visa['item_value'] ? 'selected' : '' ?>>
                                    <?= e($visa['item_label']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Visa Expiry Date</label>
                        <input type="text" class="form-control" name="visa_expiry_date" 
                               value="<?= format_uk_date($worker['visa_expiry_date']) ?>" 
                               placeholder="dd/mm/yyyy">
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Job Title</label>
                        <input type="text" class="form-control" name="job_title" 
                               value="<?= e($worker['job_title']) ?>">
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Department</label>
                        <select class="form-select" name="department">
                            <option value="">Select...</option>
                            <?php foreach ($departments as $dept): ?>
                                <option value="<?= e($dept['item_value']) ?>" 
                                        <?= $worker['department'] === $dept['item_value'] ? 'selected' : '' ?>>
                                    <?= e($dept['item_label']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Employment Start Date</label>
                        <input type="text" class="form-control" name="employment_start_date" 
                               value="<?= format_uk_date($worker['employment_start_date']) ?>" 
                               placeholder="dd/mm/yyyy">
                    </div>
                    
                    <div class="col-md-6">
                        <label class="form-label">Annual Salary (£)</label>
                        <input type="number" class="form-control" name="annual_salary" 
                               value="<?= e($worker['annual_salary']) ?>" step="0.01">
                    </div>
                    
                    <div class="col-md-12">
                        <label class="form-label">Employment Status</label>
                        <select class="form-select" name="employment_status">
                            <?php foreach ($employment_statuses as $status): ?>
                                <option value="<?= e($status['item_value']) ?>" 
                                        <?= $worker['employment_status'] === $status['item_value'] ? 'selected' : '' ?>>
                                    <?= e($status['item_label']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="card mt-4">
            <div class="card-body">
                <div class="d-flex justify-content-between">
                    <a href="<?= base_url('modules/workers/view.php?id=' . $worker_id) ?>" class="btn btn-secondary">Cancel</a>
                    <button type="submit" class="btn btn-primary btn-lg">
                        <i class="fas fa-save"></i> Update Worker
                    </button>
                </div>
            </div>
        </div>
    </form>
</div>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
