<?php
require __DIR__ . '/../../config/database.php';
require __DIR__ . '/../../includes/functions.php';

require_login();
require_permission(70); // Manager level
$page_title = 'Review Holiday Request';

$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$errors = [];

// Get request
$stmt = $pdo->prepare("
    SELECT hr.*, 
    CONCAT(sw.first_name, ' ', sw.last_name) as worker_name,
    sw.employee_number,
    sw.department,
    sw.annual_leave_entitlement
    FROM holiday_requests hr
    JOIN sponsored_workers sw ON hr.worker_id = sw.id
    WHERE hr.id = ? AND hr.business_profile_id = ? AND hr.approval_status = 'Pending'
");
$stmt->execute([$id, $_SESSION['business_profile_id']]);
$request = $stmt->fetch();

if (!$request) {
    $_SESSION['error'] = "Holiday request not found or already processed";
    header('Location: list.php');
    exit;
}

// Check for conflicting approved holidays
$stmt = $pdo->prepare("SELECT COUNT(*) FROM holiday_requests hr
                      JOIN sponsored_workers sw ON hr.worker_id = sw.id
                      WHERE sw.department = ? 
                      AND hr.approval_status = 'Approved'
                      AND hr.business_profile_id = ?
                      AND hr.id != ?
                      AND (
                          (hr.start_date <= ? AND hr.end_date >= ?) OR
                          (hr.start_date <= ? AND hr.end_date >= ?) OR
                          (hr.start_date >= ? AND hr.end_date <= ?)
                      )");
$stmt->execute([$request['department'], $_SESSION['business_profile_id'], $id,
               $request['start_date'], $request['start_date'],
               $request['end_date'], $request['end_date'],
               $request['start_date'], $request['end_date']]);
$conflicts = $stmt->fetchColumn();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $decision = $_POST['decision'] ?? '';
    $notes = trim($_POST['notes'] ?? '');
    
    if (!in_array($decision, ['approve', 'reject'])) {
        $errors[] = "Invalid decision";
    }
    
    if (empty($errors)) {
        $status = $decision === 'approve' ? 'Approved' : 'Rejected';
        
        $stmt = $pdo->prepare("UPDATE holiday_requests 
                              SET approval_status = ?,
                                  approved_by = ?,
                                  approval_date = NOW(),
                                  approval_notes = ?
                              WHERE id = ?");
        
        if ($stmt->execute([$status, $_SESSION['user_id'], $notes, $id])) {
            // Log activity
            log_activity($pdo, 'holiday_request_' . strtolower($status), 'holiday_requests', 
                       $id, "Holiday request $status for worker: {$request['worker_name']}");
            
            $_SESSION['success'] = "Holiday request $status successfully";
            header('Location: list.php');
            exit;
        } else {
            $errors[] = "Failed to update holiday request";
        }
    }
}

require __DIR__ . '/../../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-8">
        <h2><i class="bi bi-check-square"></i> Review Holiday Request</h2>
    </div>
    <div class="col-md-4 text-end">
        <a href="list.php" class="btn btn-outline-secondary">
            <i class="bi bi-arrow-left"></i> Back to List
        </a>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
        <li><?= htmlspecialchars($error) ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<?php if ($conflicts > 0): ?>
<div class="alert alert-warning">
    <i class="bi bi-exclamation-triangle"></i>
    <strong>Conflict Warning:</strong> There are <?= $conflicts ?> other approved holiday(s) 
    from the same department during this period.
</div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="stat-card">
            <h4 class="mb-4">Request Details</h4>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label text-muted">Worker</label>
                    <p class="fw-bold fs-5"><?= htmlspecialchars($request['worker_name']) ?></p>
                    <small class="text-muted"><?= htmlspecialchars($request['employee_number']) ?></small>
                </div>
                <div class="col-md-6">
                    <label class="form-label text-muted">Department</label>
                    <p><?= htmlspecialchars($request['department'] ?: 'Not specified') ?></p>
                </div>
            </div>
            
            <hr>
            
            <div class="row mb-3">
                <div class="col-md-3">
                    <label class="form-label text-muted">Leave Type</label>
                    <p><?= htmlspecialchars($request['leave_type']) ?></p>
                </div>
                <div class="col-md-3">
                    <label class="form-label text-muted">Start Date</label>
                    <p class="fw-bold"><?= date('d/m/Y', strtotime($request['start_date'])) ?></p>
                </div>
                <div class="col-md-3">
                    <label class="form-label text-muted">End Date</label>
                    <p class="fw-bold"><?= date('d/m/Y', strtotime($request['end_date'])) ?></p>
                </div>
                <div class="col-md-3">
                    <label class="form-label text-muted">Total Days</label>
                    <p class="fw-bold fs-5 text-primary"><?= $request['total_days'] ?> days</p>
                </div>
            </div>
            
            <?php if (!empty($request['reason'])): ?>
            <div class="mb-3">
                <label class="form-label text-muted">Worker's Reason</label>
                <div class="alert alert-light">
                    <?= nl2br(htmlspecialchars($request['reason'])) ?>
                </div>
            </div>
            <?php endif; ?>
            
            <hr>
            
            <!-- Decision Form -->
            <form method="POST" action="">
                <h5 class="mb-3">Your Decision</h5>
                
                <div class="mb-3">
                    <label class="form-label">Manager Notes (optional)</label>
                    <textarea name="notes" class="form-control" rows="3" 
                              placeholder="Add any notes about your decision..."></textarea>
                    <small class="text-muted">These notes will be visible to HR and the worker</small>
                </div>
                
                <div class="d-flex gap-2">
                    <button type="submit" name="decision" value="approve" class="btn btn-success btn-lg">
                        <i class="bi bi-check-circle"></i> Approve Request
                    </button>
                    <button type="submit" name="decision" value="reject" class="btn btn-danger btn-lg">
                        <i class="bi bi-x-circle"></i> Reject Request
                    </button>
                    <a href="list.php" class="btn btn-outline-secondary btn-lg">Cancel</a>
                </div>
            </form>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="stat-card">
            <h6><i class="bi bi-info-circle"></i> Leave Entitlement</h6>
            <p class="mb-0">Annual Entitlement: <strong><?= $request['annual_leave_entitlement'] ?> days</strong></p>
            <small class="text-muted">Full year entitlement for this worker</small>
        </div>
        
        <?php if ($conflicts > 0): ?>
        <div class="stat-card mt-3 border-warning">
            <h6 class="text-warning"><i class="bi bi-exclamation-triangle"></i> Department Conflicts</h6>
            <p class="mb-0"><?= $conflicts ?> team member(s) already on holiday during this period</p>
            <small class="text-muted">Consider department coverage before approving</small>
        </div>
        <?php endif; ?>
        
        <div class="stat-card mt-3">
            <h6><i class="bi bi-calendar3"></i> Request Details</h6>
            <div class="d-flex justify-content-between mb-2">
                <span>Requested:</span>
                <small><?= date('d/m/Y H:i', strtotime($request['created_at'])) ?></small>
            </div>
            <div class="d-flex justify-content-between">
                <span>Working Days:</span>
                <strong class="text-primary"><?= $request['total_days'] ?></strong>
            </div>
        </div>
    </div>
</div>

<?php require __DIR__ . '/../../includes/footer.php'; ?>
