<?php
$page_title = 'System Settings';
require_once '../../includes/header.php';
require_permission(80); // Admin only

$success = '';
$errors = [];
$tab = $_GET['tab'] ?? 'business';

// Fetch business profile
$stmt = $pdo->prepare("SELECT * FROM business_profiles WHERE id = ?");
$stmt->execute([$_SESSION['business_profile_id']]);
$business = $stmt->fetch();

// Handle business profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_business'])) {
    try {
        $stmt = $pdo->prepare("UPDATE business_profiles SET 
                              business_name = ?, trading_name = ?, registration_number = ?, 
                              vat_number = ?, address_line1 = ?, address_line2 = ?, 
                              city = ?, postcode = ?, phone = ?, email = ?, website = ?, 
                              sponsor_license_number = ?, license_issue_date = ?, 
                              license_expiry_date = ? 
                              WHERE id = ?");
        $stmt->execute([
            $_POST['business_name'],
            $_POST['trading_name'] ?: null,
            $_POST['registration_number'] ?: null,
            $_POST['vat_number'] ?: null,
            $_POST['address_line1'] ?: null,
            $_POST['address_line2'] ?: null,
            $_POST['city'] ?: null,
            $_POST['postcode'] ?: null,
            $_POST['phone'] ?: null,
            $_POST['email'] ?: null,
            $_POST['website'] ?: null,
            $_POST['sponsor_license_number'] ?: null,
            $_POST['license_issue_date'] ?: null,
            $_POST['license_expiry_date'] ?: null,
            $_SESSION['business_profile_id']
        ]);
        
        log_action($pdo, 'Business Profile Updated', 'business_profiles', $_SESSION['business_profile_id']);
        $success = 'Business profile updated successfully';
        
        // Refresh data
        $stmt = $pdo->prepare("SELECT * FROM business_profiles WHERE id = ?");
        $stmt->execute([$_SESSION['business_profile_id']]);
        $business = $stmt->fetch();
        
    } catch (PDOException $e) {
        $errors[] = "Database error: " . $e->getMessage();
    }
}

// Handle add list item
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_list_item'])) {
    try {
        $stmt = $pdo->prepare("INSERT INTO system_lists 
                              (business_profile_id, list_type, item_value, item_label, is_active) 
                              VALUES (?, ?, ?, ?, 1)");
        $stmt->execute([
            $_SESSION['business_profile_id'],
            $_POST['list_type'],
            strtolower(str_replace(' ', '_', $_POST['item_value'])),
            $_POST['item_label']
        ]);
        
        $success = 'List item added successfully';
    } catch (PDOException $e) {
        $errors[] = "Error: " . $e->getMessage();
    }
}

// Get all list items
$stmt = $pdo->query("SELECT * FROM system_lists WHERE is_system = 0 OR business_profile_id IS NOT NULL ORDER BY list_type, display_order");
$list_items = $stmt->fetchAll();
$grouped_lists = [];
foreach ($list_items as $item) {
    $grouped_lists[$item['list_type']][] = $item;
}
?>

<?php if ($success): echo show_alert($success, 'success'); endif; ?>
<?php if (!empty($errors)): ?>
    <div class="alert alert-danger">
        <?php foreach ($errors as $error): echo htmlspecialchars($error) . '<br>'; endforeach; ?>
    </div>
<?php endif; ?>

<!-- Tabs -->
<ul class="nav nav-tabs mb-3">
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'business' ? 'active' : '' ?>" href="?tab=business">
            <i class="bi bi-building"></i> Business Profile
        </a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'lists' ? 'active' : '' ?>" href="?tab=lists">
            <i class="bi bi-list-ul"></i> System Lists
        </a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'users' ? 'active' : '' ?>" href="?tab=users">
            <i class="bi bi-people"></i> Users
        </a>
    </li>
    <?php if ($_SESSION['permission_level'] >= 100): ?>
    <li class="nav-item">
        <a class="nav-link" href="module_manager.php">
            <i class="bi bi-puzzle"></i> Module Manager
        </a>
    </li>
    <?php endif; ?>
</ul>

<?php if ($tab === 'business'): ?>
<!-- Business Profile Tab -->
<div class="stat-card">
    <h5><i class="bi bi-building"></i> Business Profile</h5>
    <hr>
    
    <form method="POST">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-primary mb-3">Company Information</h6>
                
                <div class="mb-3">
                    <label class="form-label">Business Name <span class="text-danger">*</span></label>
                    <input type="text" name="business_name" class="form-control" required 
                           value="<?= htmlspecialchars($business['business_name']) ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Trading Name</label>
                    <input type="text" name="trading_name" class="form-control" 
                           value="<?= htmlspecialchars($business['trading_name'] ?? '') ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Registration Number</label>
                    <input type="text" name="registration_number" class="form-control" 
                           value="<?= htmlspecialchars($business['registration_number'] ?? '') ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">VAT Number</label>
                    <input type="text" name="vat_number" class="form-control" 
                           value="<?= htmlspecialchars($business['vat_number'] ?? '') ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Address Line 1</label>
                    <input type="text" name="address_line1" class="form-control" 
                           value="<?= htmlspecialchars($business['address_line1'] ?? '') ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Address Line 2</label>
                    <input type="text" name="address_line2" class="form-control" 
                           value="<?= htmlspecialchars($business['address_line2'] ?? '') ?>">
                </div>
                
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label">City</label>
                        <input type="text" name="city" class="form-control" 
                               value="<?= htmlspecialchars($business['city'] ?? '') ?>">
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Postcode</label>
                        <input type="text" name="postcode" class="form-control" 
                               value="<?= htmlspecialchars($business['postcode'] ?? '') ?>">
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <h6 class="text-primary mb-3">Contact & License Information</h6>
                
                <div class="mb-3">
                    <label class="form-label">Phone</label>
                    <input type="tel" name="phone" class="form-control" 
                           value="<?= htmlspecialchars($business['phone'] ?? '') ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Email</label>
                    <input type="email" name="email" class="form-control" 
                           value="<?= htmlspecialchars($business['email'] ?? '') ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Website</label>
                    <input type="url" name="website" class="form-control" 
                           value="<?= htmlspecialchars($business['website'] ?? '') ?>">
                </div>
                
                <hr class="my-4">
                
                <div class="mb-3">
                    <label class="form-label">Sponsor License Number</label>
                    <input type="text" name="sponsor_license_number" class="form-control" 
                           value="<?= htmlspecialchars($business['sponsor_license_number'] ?? '') ?>">
                </div>
                
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label">License Issue Date</label>
                        <input type="date" name="license_issue_date" class="form-control" 
                               value="<?= $business['license_issue_date'] ?>">
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="form-label">License Expiry Date</label>
                        <input type="date" name="license_expiry_date" class="form-control" 
                               value="<?= $business['license_expiry_date'] ?>">
                    </div>
                </div>
            </div>
        </div>
        
        <hr>
        
        <button type="submit" name="update_business" class="btn btn-primary">
            <i class="bi bi-save"></i> Save Changes
        </button>
    </form>
</div>

<?php elseif ($tab === 'lists'): ?>
<!-- System Lists Tab -->
<div class="stat-card">
    <h5><i class="bi bi-list-ul"></i> System Lists Management</h5>
    <hr>
    
    <!-- Add New Item Form -->
    <div class="card bg-light mb-4">
        <div class="card-body">
            <h6>Add New List Item</h6>
            <form method="POST" class="row g-3">
                <div class="col-md-4">
                    <select name="list_type" class="form-select" required>
                        <option value="">Select List Type...</option>
                        <option value="departments">Departments</option>
                        <option value="visa_types">Visa Types</option>
                        <option value="contract_types">Contract Types</option>
                        <option value="work_patterns">Work Patterns</option>
                        <option value="leave_types">Leave Types</option>
                        <option value="document_categories">Document Categories</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <input type="text" name="item_value" class="form-control" 
                           placeholder="Value (e.g., sales)" required>
                </div>
                <div class="col-md-3">
                    <input type="text" name="item_label" class="form-control" 
                           placeholder="Label (e.g., Sales)" required>
                </div>
                <div class="col-md-2">
                    <button type="submit" name="add_list_item" class="btn btn-success w-100">
                        <i class="bi bi-plus"></i> Add
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Display Lists -->
    <?php foreach ($grouped_lists as $list_type => $items): ?>
    <div class="mb-4">
        <h6 class="text-primary"><?= ucwords(str_replace('_', ' ', $list_type)) ?></h6>
        <div class="table-responsive">
            <table class="table table-sm table-hover">
                <thead>
                    <tr>
                        <th>Value</th>
                        <th>Label</th>
                        <th>Status</th>
                        <th>System</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($items as $item): ?>
                    <tr>
                        <td><code><?= htmlspecialchars($item['item_value']) ?></code></td>
                        <td><?= htmlspecialchars($item['item_label']) ?></td>
                        <td>
                            <span class="badge <?= $item['is_active'] ? 'bg-success' : 'bg-secondary' ?>">
                                <?= $item['is_active'] ? 'Active' : 'Inactive' ?>
                            </span>
                        </td>
                        <td>
                            <?= $item['is_system'] ? '<span class="badge bg-info">System</span>' : '<span class="badge bg-secondary">Custom</span>' ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<?php elseif ($tab === 'users'): ?>
<!-- Users Tab -->
<div class="stat-card">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5><i class="bi bi-people"></i> System Users</h5>
        <a href="add_user.php" class="btn btn-primary btn-sm">
            <i class="bi bi-person-plus"></i> Add User
        </a>
    </div>
    
    <?php
    $stmt = $pdo->prepare("SELECT * FROM users WHERE business_profile_id = ? ORDER BY permission_level DESC, full_name");
    $stmt->execute([$_SESSION['business_profile_id']]);
    $users = $stmt->fetchAll();
    ?>
    
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>Name</th>
                    <th>Username</th>
                    <th>Email</th>
                    <th>Role</th>
                    <th>Permission Level</th>
                    <th>Status</th>
                    <th>Last Login</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($users as $user): ?>
                <tr>
                    <td><strong><?= htmlspecialchars($user['full_name']) ?></strong></td>
                    <td><?= htmlspecialchars($user['username']) ?></td>
                    <td><?= htmlspecialchars($user['email']) ?></td>
                    <td><?= htmlspecialchars($user['user_role']) ?></td>
                    <td>
                        <span class="badge bg-<?= $user['permission_level'] >= 80 ? 'danger' : ($user['permission_level'] >= 60 ? 'warning' : 'primary') ?>">
                            Level <?= $user['permission_level'] ?>
                        </span>
                    </td>
                    <td>
                        <span class="badge <?= $user['is_active'] ? 'bg-success' : 'bg-secondary' ?>">
                            <?= $user['is_active'] ? 'Active' : 'Inactive' ?>
                        </span>
                    </td>
                    <td><?= $user['last_login'] ? format_datetime($user['last_login']) : '<span class="text-muted">Never</span>' ?></td>
                    <td>
                        <?php if ($user['id'] !== $_SESSION['user_id']): ?>
                        <a href="edit_user.php?id=<?= $user['id'] ?>" class="btn btn-sm btn-outline-warning" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </a>
                        <?php else: ?>
                        <span class="badge bg-info">You</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    <div class="alert alert-info mt-3">
        <strong><i class="bi bi-info-circle"></i> Permission Levels:</strong>
        <ul class="mb-0 small mt-2">
            <li><strong>100:</strong> System Administrator - Full access to everything</li>
            <li><strong>90:</strong> HR Director - All HR functions including user management</li>
            <li><strong>80:</strong> HR Manager - Worker and document management</li>
            <li><strong>70:</strong> Department Manager - Team management and approvals</li>
            <li><strong>60:</strong> Supervisor - Attendance marking and basic reporting</li>
            <li><strong>50:</strong> Staff - View own records only</li>
            <li><strong>30-40:</strong> Limited/Read-only access</li>
        </ul>
    </div>
</div>
<?php endif; ?>

<?php require_once '../../includes/footer.php'; ?>