<?php
/**
 * EDIT USER MODULE
 * SAVE AS: modules/admin/edit_user.php
 * Edit existing users with permission control
 */

$page_title = 'Edit User';
require_once '../../includes/header.php';
require_permission(90);

$user_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$errors = [];
$success = '';

// Fetch user
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND business_profile_id = ?");
$stmt->execute([$user_id, $_SESSION['business_profile_id']]);
$user = $stmt->fetch();

if (!$user) {
    header('Location: system_settings.php?tab=users&error=User not found');
    exit;
}

// Cannot edit users with higher permissions
if ($user['permission_level'] > $_SESSION['permission_level']) {
    header('Location: system_settings.php?tab=users&error=Cannot edit user with higher permissions');
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $required = ['full_name', 'email', 'user_role', 'permission_level'];
    
    foreach ($required as $field) {
        if (empty($_POST[$field])) {
            $errors[] = ucwords(str_replace('_', ' ', $field)) . " is required";
        }
    }
    
    // Check permission level
    $requested_level = (int)$_POST['permission_level'];
    if ($requested_level > $_SESSION['permission_level']) {
        $errors[] = "You cannot assign higher permissions than your own";
    }
    
    // Handle password change
    if (!empty($_POST['new_password'])) {
        if ($_POST['new_password'] !== $_POST['confirm_password']) {
            $errors[] = "Passwords do not match";
        }
        if (strlen($_POST['new_password']) < 8) {
            $errors[] = "Password must be at least 8 characters";
        }
    }
    
    if (empty($errors)) {
        try {
            // Update user details
            $sql = "UPDATE users SET 
                    full_name = ?, 
                    email = ?, 
                    user_role = ?, 
                    permission_level = ?, 
                    phone = ?, 
                    is_active = ?";
            
            $params = [
                $_POST['full_name'],
                $_POST['email'],
                $_POST['user_role'],
                $requested_level,
                $_POST['phone'] ?? null,
                isset($_POST['is_active']) ? 1 : 0
            ];
            
            // Add password if provided
            if (!empty($_POST['new_password'])) {
                $sql .= ", password = ?, password_changed_at = NOW()";
                $params[] = password_hash($_POST['new_password'], PASSWORD_DEFAULT);
            }
            
            $sql .= " WHERE id = ? AND business_profile_id = ?";
            $params[] = $user_id;
            $params[] = $_SESSION['business_profile_id'];
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            
            log_action($pdo, 'User Updated', 'users', $user_id);
            
            header('Location: system_settings.php?tab=users&success=User updated successfully');
            exit;
            
        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

$role_presets = [
    100 => ['name' => 'System Administrator', 'description' => 'Full access to everything'],
    90 => ['name' => 'HR Director', 'description' => 'All HR functions + User Management'],
    80 => ['name' => 'HR Manager', 'description' => 'Worker & Document Management'],
    70 => ['name' => 'Department Manager', 'description' => 'Team Management & Approvals'],
    60 => ['name' => 'Supervisor', 'description' => 'Attendance & Basic Reporting'],
    50 => ['name' => 'Staff', 'description' => 'View Own Records Only'],
    40 => ['name' => 'Read Only', 'description' => 'View Only Access'],
    30 => ['name' => 'Limited', 'description' => 'Basic Info Only']
];
?>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger">
        <ul class="mb-0">
            <?php foreach ($errors as $error): ?>
                <li><?= htmlspecialchars($error) ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<div class="stat-card">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5><i class="bi bi-pencil"></i> Edit User: <?= htmlspecialchars($user['full_name']) ?></h5>
        <a href="system_settings.php?tab=users" class="btn btn-outline-secondary btn-sm">
            <i class="bi bi-arrow-left"></i> Back to Users
        </a>
    </div>
    <hr>
    
    <form method="POST">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-primary mb-3">Account Information</h6>
                
                <div class="mb-3">
                    <label class="form-label">Full Name <span class="text-danger">*</span></label>
                    <input type="text" name="full_name" class="form-control" required 
                           value="<?= htmlspecialchars($user['full_name']) ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Email <span class="text-danger">*</span></label>
                    <input type="email" name="email" class="form-control" required 
                           value="<?= htmlspecialchars($user['email']) ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Phone</label>
                    <input type="tel" name="phone" class="form-control" 
                           value="<?= htmlspecialchars($user['phone'] ?? '') ?>">
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Username</label>
                    <input type="text" class="form-control" 
                           value="<?= htmlspecialchars($user['username']) ?>" disabled>
                    <small class="text-muted">Username cannot be changed</small>
                </div>
                
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" name="is_active" id="is_active" 
                           <?= $user['is_active'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="is_active">
                        Account Active
                    </label>
                </div>
            </div>
            
            <div class="col-md-6">
                <h6 class="text-primary mb-3">Permissions & Role</h6>
                
                <div class="mb-3">
                    <label class="form-label">Permission Level <span class="text-danger">*</span></label>
                    <select name="permission_level" id="permission_level" class="form-select" required>
                        <?php foreach ($role_presets as $level => $preset): ?>
                            <?php if ($level <= $_SESSION['permission_level']): ?>
                            <option value="<?= $level ?>" <?= $user['permission_level'] == $level ? 'selected' : '' ?>>
                                Level <?= $level ?> - <?= htmlspecialchars($preset['name']) ?>
                            </option>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Role Title <span class="text-danger">*</span></label>
                    <input type="text" name="user_role" id="user_role" class="form-control" required 
                           value="<?= htmlspecialchars($user['user_role']) ?>">
                </div>
                
                <hr>
                
                <h6 class="text-primary mb-3">Change Password (Optional)</h6>
                
                <div class="mb-3">
                    <label class="form-label">New Password</label>
                    <input type="password" name="new_password" class="form-control" minlength="8">
                    <small class="text-muted">Leave blank to keep current password</small>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Confirm New Password</label>
                    <input type="password" name="confirm_password" class="form-control" minlength="8">
                </div>
            </div>
        </div>
        
        <hr>
        
        <div class="row">
            <div class="col-md-6">
                <div class="card bg-light">
                    <div class="card-body">
                        <h6>Account Information</h6>
                        <table class="table table-sm mb-0">
                            <tr>
                                <td>Created:</td>
                                <td><?= format_datetime($user['created_at']) ?></td>
                            </tr>
                            <tr>
                                <td>Last Login:</td>
                                <td><?= $user['last_login'] ? format_datetime($user['last_login']) : 'Never' ?></td>
                            </tr>
                            <tr>
                                <td>Password Changed:</td>
                                <td><?= $user['password_changed_at'] ? format_datetime($user['password_changed_at']) : 'Never' ?></td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="alert alert-info mb-0">
                    <h6>Editing Rules</h6>
                    <ul class="small mb-0">
                        <li>You cannot edit users with higher permissions than yours</li>
                        <li>You cannot assign permissions higher than yours</li>
                        <li>Username cannot be changed</li>
                        <li>Deactivating account will prevent login</li>
                        <li>Password is only changed if you fill in the fields</li>
                    </ul>
                </div>
            </div>
        </div>
        
        <hr>
        
        <div class="d-flex justify-content-between">
            <a href="system_settings.php?tab=users" class="btn btn-secondary">
                <i class="bi bi-x-circle"></i> Cancel
            </a>
            <button type="submit" class="btn btn-primary">
                <i class="bi bi-save"></i> Save Changes
            </button>
        </div>
    </form>
</div>

<script>
// Auto-fill role based on permission level
document.getElementById('permission_level').addEventListener('change', function() {
    const level = parseInt(this.value);
    const roleInput = document.getElementById('user_role');
    
    const presets = <?= json_encode($role_presets) ?>;
    
    if (presets[level]) {
        roleInput.value = presets[level].name;
    }
});
</script>

<?php require_once '../../includes/footer.php'; ?>