<?php
/**
 * Step 1 Remote: Download Files from Server
 * Only shown if install_mode = 'remote'
 */

// Check if remote install mode
if (!isset($_SESSION['install_mode']) || $_SESSION['install_mode'] !== 'remote') {
    header('Location: install.php?step=2');
    exit;
}

// RIU Server Configuration
define('RIU_SERVER', 'https://supix.co.uk/sys/riu');  // RIU Control Plane
define('PRODUCT_CODE', 'scms');
define('PRODUCT_ID', 'scms_v3');
?>

<h3 class="mb-4"><i class="bi bi-cloud-download-fill"></i> Downloading Files from Server</h3>

<div class="alert alert-info">
    <i class="bi bi-info-circle-fill"></i> <strong>Remote Installation</strong><br>
    Files are being downloaded from our deployment server. This may take a few minutes depending on your internet connection.
</div>

<!-- Progress Container -->
<div class="card">
    <div class="card-body">
        <h5 class="mb-4">Installation Progress</h5>

        <!-- Overall Progress -->
        <div class="mb-4">
            <div class="d-flex justify-content-between mb-2">
                <span><strong>Overall Progress</strong></span>
                <span id="overall_percent">0%</span>
            </div>
            <div class="progress" style="height: 30px;">
                <div class="progress-bar progress-bar-striped progress-bar-animated bg-success" 
                     role="progressbar" 
                     id="overall_progress" 
                     style="width: 0%"
                     aria-valuenow="0" 
                     aria-valuemin="0" 
                     aria-valuemax="100">
                    0%
                </div>
            </div>
        </div>

        <!-- Current Task -->
        <div class="mb-4">
            <div class="d-flex justify-content-between mb-2">
                <span><strong>Current Task:</strong></span>
                <span id="current_task_status" class="text-muted">Initializing...</span>
            </div>
            <p id="current_task" class="text-primary mb-0">
                <i class="bi bi-hourglass-split"></i> Connecting to deployment server...
            </p>
        </div>

        <!-- Detailed Progress Log -->
        <div class="card bg-light">
            <div class="card-header">
                <strong>Installation Log</strong>
                <button type="button" class="btn btn-sm btn-outline-secondary float-end" onclick="toggleLog()">
                    <i class="bi bi-eye" id="log_icon"></i> <span id="log_toggle_text">Show Details</span>
                </button>
            </div>
            <div class="card-body p-0" id="log_container" style="display: none;">
                <div id="install_log" style="max-height: 300px; overflow-y: auto; font-family: monospace; font-size: 12px; padding: 15px; background: #f8f9fa;">
                    <div class="text-muted">Starting installation...</div>
                </div>
            </div>
        </div>

        <!-- File Statistics -->
        <div class="row mt-4 text-center">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="total_files">0</h4>
                        <small>Total Files</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="downloaded_files">0</h4>
                        <small>Downloaded</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-info text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="download_size">0 MB</h4>
                        <small>Downloaded</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="eta">--:--</h4>
                        <small>ETA</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Success/Error Messages -->
<div id="install_complete" class="alert alert-success mt-4" style="display: none;">
    <i class="bi bi-check-circle-fill"></i> <strong>Download Complete!</strong><br>
    All files have been successfully downloaded. Proceeding to database configuration...
</div>

<div id="install_error" class="alert alert-danger mt-4" style="display: none;">
    <i class="bi bi-x-circle-fill"></i> <strong>Download Failed!</strong><br>
    <span id="error_message"></span>
</div>

<script>
let installLog = [];
let startTime = Date.now();
let totalFiles = 0;
let downloadedFiles = 0;
let downloadedSize = 0;

function toggleLog() {
    const logContainer = document.getElementById('log_container');
    const toggleText = document.getElementById('log_toggle_text');
    const icon = document.getElementById('log_icon');
    
    if (logContainer.style.display === 'none') {
        logContainer.style.display = 'block';
        toggleText.textContent = 'Hide Details';
        icon.className = 'bi bi-eye-slash';
    } else {
        logContainer.style.display = 'none';
        toggleText.textContent = 'Show Details';
        icon.className = 'bi bi-eye';
    }
}

function addLog(message, type = 'info') {
    const timestamp = new Date().toLocaleTimeString();
    const logDiv = document.getElementById('install_log');
    const colors = {
        'info': 'text-muted',
        'success': 'text-success',
        'error': 'text-danger',
        'warning': 'text-warning'
    };
    
    const logEntry = `<div class="${colors[type]}">[${timestamp}] ${message}</div>`;
    logDiv.innerHTML += logEntry;
    logDiv.scrollTop = logDiv.scrollHeight;
}

function updateProgress(percent, task, status = '') {
    // Update overall progress bar
    document.getElementById('overall_progress').style.width = percent + '%';
    document.getElementById('overall_progress').textContent = Math.round(percent) + '%';
    document.getElementById('overall_percent').textContent = Math.round(percent) + '%';
    
    // Update current task
    document.getElementById('current_task').innerHTML = '<i class="bi bi-gear-fill spinner-grow spinner-grow-sm"></i> ' + task;
    if (status) {
        document.getElementById('current_task_status').textContent = status;
    }
}

function updateStats(total, downloaded, size) {
    document.getElementById('total_files').textContent = total;
    document.getElementById('downloaded_files').textContent = downloaded;
    document.getElementById('download_size').textContent = (size / 1024 / 1024).toFixed(2) + ' MB';
    
    // Calculate ETA
    const elapsed = (Date.now() - startTime) / 1000;
    const rate = downloaded / elapsed;
    const remaining = total - downloaded;
    const eta = remaining / rate;
    
    const minutes = Math.floor(eta / 60);
    const seconds = Math.floor(eta % 60);
    document.getElementById('eta').textContent = minutes + ':' + (seconds < 10 ? '0' : '') + seconds;
}

// Start remote installation
async function startRemoteInstall() {
    try {
        addLog('Connecting to RIU server: <?php echo RIU_SERVER; ?>', 'info');
        updateProgress(5, 'Authenticating with license server...');

        // Step 1: Verify license with RIU and get access token
        const licenseData = {
            license_key: '<?php echo $_SESSION["license_key"]; ?>',
            domain: '<?php echo $_SESSION["domain"]; ?>',
            product_id: '<?php echo PRODUCT_ID; ?>'
        };

        addLog('Verifying license: ' + licenseData.license_key, 'info');

        const verifyResponse = await fetch('<?php echo RIU_SERVER; ?>/api/license/verify', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify(licenseData)
        });

        if (!verifyResponse.ok) {
            throw new Error('Failed to connect to RIU server');
        }

        const verifyResult = await verifyResponse.json();

        if (!verifyResult.valid) {
            throw new Error(verifyResult.message || 'License verification failed');
        }

        addLog('License verified successfully', 'success');
        const accessToken = verifyResult.access_token;
        updateProgress(10, 'License verified, fetching manifest...');

        // Step 2: Get file manifest from RIU
        addLog('Requesting file manifest...', 'info');

        const manifestResponse = await fetch('<?php echo RIU_SERVER; ?>/api/distribution/manifest/<?php echo PRODUCT_ID; ?>/latest', {
            method: 'GET',
            headers: {
                'Authorization': 'Bearer ' + accessToken,
                'Content-Type': 'application/json'
            }
        });
        
        if (!manifestResponse.ok) {
            throw new Error('Failed to fetch manifest from RIU');
        }

        const manifest = await manifestResponse.json();

        if (!manifest || !manifest.files) {
            throw new Error('Invalid manifest received from RIU');
        }

        addLog('Manifest received successfully', 'success');
        addLog('Product: ' + manifest.product_id + ' v' + manifest.version, 'info');
        addLog('Files to download: ' + manifest.files.length, 'info');
        updateProgress(10, 'Downloading file list...', 'Connected');
        
        totalFiles = manifest.files.length;
        updateStats(totalFiles, 0, 0);
        
        // Step 2: Create directory structure
        addLog('Creating directory structure...', 'info');
        updateProgress(15, 'Creating directories...');
        
        const createDirsResponse = await fetch('install_ajax.php?action=create_directories', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({directories: manifest.directories})
        });
        
        const dirsResult = await createDirsResponse.json();
        if (!dirsResult.success) {
            throw new Error('Failed to create directories: ' + dirsResult.message);
        }
        
        addLog('Created ' + manifest.directories.length + ' directories', 'success');
        updateProgress(20, 'Directories created successfully');
        
        // Step 3: Download files
        addLog('Starting file downloads...', 'info');
        let downloadedCount = 0;
        let downloadedBytes = 0;
        
        for (let i = 0; i < manifest.files.length; i++) {
            const file = manifest.files[i];
            const progress = 20 + ((i / manifest.files.length) * 70);
            
            updateProgress(progress, `Downloading: ${file.path}`, `${i + 1}/${totalFiles}`);
            addLog('Downloading: ' + file.path + ' (' + (file.size / 1024).toFixed(1) + ' KB)', 'info');

            // Download file from RIU
            const fileResponse = await fetch('<?php echo RIU_SERVER; ?>/api/distribution/download/<?php echo PRODUCT_ID; ?>/' + manifest.version + '/' + encodeURIComponent(file.path), {
                method: 'GET',
                headers: {
                    'Authorization': 'Bearer ' + accessToken
                }
            });
            
            if (!fileResponse.ok) {
                throw new Error('Failed to download: ' + file.path);
            }
            
            const fileBlob = await fileResponse.blob();
            
            // Save file via AJAX
            const formData = new FormData();
            formData.append('file_path', file.path);
            formData.append('file_content', fileBlob);
            
            const saveResponse = await fetch('install_ajax.php?action=save_file', {
                method: 'POST',
                body: formData
            });
            
            const saveResult = await saveResponse.json();
            if (!saveResult.success) {
                throw new Error('Failed to save: ' + file.path);
            }
            
            downloadedCount++;
            downloadedBytes += file.size;
            updateStats(totalFiles, downloadedCount, downloadedBytes);
            
            // Small delay to prevent server overload
            if (i % 10 === 0) {
                await new Promise(resolve => setTimeout(resolve, 100));
            }
        }
        
        addLog('All files downloaded successfully!', 'success');
        updateProgress(95, 'Finalizing installation...');
        
        // Step 4: Verify installation
        addLog('Verifying file integrity...', 'info');
        const verifyResponse = await fetch('install_ajax.php?action=verify_installation', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({checksum: manifest.checksum})
        });
        
        const verifyResult = await verifyResponse.json();
        if (!verifyResult.success) {
            addLog('Warning: Some files may be corrupted', 'warning');
        } else {
            addLog('File integrity verified', 'success');
        }
        
        updateProgress(100, 'Download complete!', 'Success');
        
        // Show success message
        document.getElementById('install_complete').style.display = 'block';
        addLog('Installation files ready. Redirecting to database configuration...', 'success');
        
        // Redirect to next step after 3 seconds
        setTimeout(() => {
            window.location.href = 'install.php?step=3';
        }, 3000);
        
    } catch (error) {
        addLog('ERROR: ' + error.message, 'error');
        document.getElementById('install_error').style.display = 'block';
        document.getElementById('error_message').textContent = error.message;
        updateProgress(0, 'Installation failed');
    }
}

// Auto-start installation when page loads
window.addEventListener('load', function() {
    setTimeout(startRemoteInstall, 500);
});
</script>
