<?php
/**
 * Step 1 Local: Verify Files on Local Server
 * Only shown if install_mode = 'local'
 */

// Check if local install mode
if (!isset($_SESSION['install_mode']) || $_SESSION['install_mode'] !== 'local') {
    header('Location: install.php?step=2');
    exit;
}

// Define expected files for SCMS v3.0
$required_files = array(
    // Core files
    'includes/core.php',
    'includes/helpers.php',
    'includes/database.php',
    'config/database.sample.php',
    
    // Admin files
    'admin/login.php',
    'admin/dashboard.php',
    'admin/header.php',
    'admin/footer.php',
    'admin/sidebar.php',
    
    // Worker module
    'admin/workers/index.php',
    'admin/workers/add.php',
    'admin/workers/edit.php',
    'admin/workers/view.php',
    'admin/workers/delete.php',
    
    // Attendance module
    'admin/attendance/index.php',
    'admin/attendance/calendar.php',
    'admin/attendance/reports.php',
    
    // Holiday module
    'admin/holidays/index.php',
    'admin/holidays/requests.php',
    'admin/holidays/approve.php',
    
    // Document module
    'admin/documents/index.php',
    'admin/documents/upload.php',
    'admin/documents/expiry_alerts.php',
    
    // CoS module
    'admin/cos/index.php',
    'admin/cos/assign.php',
    'admin/cos/reports.php',
    
    // Home Office module
    'admin/home_office/index.php',
    'admin/home_office/reports.php',
    'admin/home_office/submit.php',
    
    // RTW module
    'admin/rtw/index.php',
    'admin/rtw/checks.php',
    
    // Reports
    'admin/reports/index.php',
    'admin/reports/workers.php',
    'admin/reports/attendance.php',
    'admin/reports/compliance.php',
    
    // Settings
    'admin/settings/index.php',
    'admin/settings/business.php',
    'admin/settings/system.php',
    'admin/settings/users.php',
    
    // Assets
    'assets/css/style.css',
    'assets/js/main.js',
    'assets/images/logo.png'
);

$total_files = count($required_files);
?>

<h3 class="mb-4"><i class="bi bi-hdd-fill"></i> Verifying Local Files</h3>

<div class="alert alert-info">
    <i class="bi bi-info-circle-fill"></i> <strong>Local Installation</strong><br>
    Checking for required files on your server. This will verify that all SCMS files are present and valid.
</div>

<!-- Progress Container -->
<div class="card">
    <div class="card-body">
        <h5 class="mb-4">File Verification Progress</h5>

        <!-- Overall Progress -->
        <div class="mb-4">
            <div class="d-flex justify-content-between mb-2">
                <span><strong>Verification Progress</strong></span>
                <span id="overall_percent">0%</span>
            </div>
            <div class="progress" style="height: 30px;">
                <div class="progress-bar progress-bar-striped progress-bar-animated bg-info" 
                     role="progressbar" 
                     id="overall_progress" 
                     style="width: 0%">
                    0%
                </div>
            </div>
        </div>

        <!-- Current File -->
        <div class="mb-4">
            <div class="d-flex justify-content-between mb-2">
                <span><strong>Current File:</strong></span>
                <span id="current_status" class="text-muted">Starting...</span>
            </div>
            <p id="current_file" class="text-primary mb-0" style="font-family: monospace; font-size: 14px;">
                <i class="bi bi-hourglass-split"></i> Initializing verification...
            </p>
        </div>

        <!-- File Statistics -->
        <div class="row mb-4 text-center">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="total_files"><?php echo $total_files; ?></h4>
                        <small>Total Files</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="verified_files">0</h4>
                        <small>Verified</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-danger text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="missing_files">0</h4>
                        <small>Missing</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="total_size">0 MB</h4>
                        <small>Total Size</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Verification Log -->
        <div class="card bg-light">
            <div class="card-header">
                <strong>Verification Details</strong>
                <button type="button" class="btn btn-sm btn-outline-secondary float-end" onclick="toggleLog()">
                    <i class="bi bi-eye" id="log_icon"></i> <span id="log_toggle_text">Show Details</span>
                </button>
            </div>
            <div class="card-body p-0" id="log_container" style="display: none;">
                <div id="verify_log" style="max-height: 300px; overflow-y: auto; font-family: 'Courier New', monospace; font-size: 12px; padding: 15px; background: #f8f9fa;">
                    <div class="text-muted">[Starting verification...]</div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Success Message -->
<div id="verify_complete" class="alert alert-success mt-4" style="display: none;">
    <i class="bi bi-check-circle-fill"></i> <strong>Verification Complete!</strong><br>
    All required files are present. Proceeding to database configuration...
</div>

<!-- Warning Message (missing files) -->
<div id="verify_warning" class="alert alert-warning mt-4" style="display: none;">
    <i class="bi bi-exclamation-triangle-fill"></i> <strong>Warning: Missing Files Detected</strong><br>
    <span id="warning_message"></span>
    <div id="missing_list" class="mt-2"></div>
    <button type="button" class="btn btn-warning btn-sm mt-2" onclick="continueAnyway()">
        Continue Anyway (Not Recommended)
    </button>
</div>

<!-- Error Message -->
<div id="verify_error" class="alert alert-danger mt-4" style="display: none;">
    <i class="bi bi-x-circle-fill"></i> <strong>Critical Files Missing!</strong><br>
    <span id="error_message"></span>
</div>

<script>
// File list to verify
const filesToVerify = <?php echo json_encode($required_files); ?>;
const totalFiles = filesToVerify.length;
let verifiedCount = 0;
let missingCount = 0;
let totalSize = 0;
let missingFiles = [];

function toggleLog() {
    const logContainer = document.getElementById('log_container');
    const toggleText = document.getElementById('log_toggle_text');
    const icon = document.getElementById('log_icon');
    
    if (logContainer.style.display === 'none') {
        logContainer.style.display = 'block';
        toggleText.textContent = 'Hide Details';
        icon.className = 'bi bi-eye-slash';
    } else {
        logContainer.style.display = 'none';
        toggleText.textContent = 'Show Details';
        icon.className = 'bi bi-eye';
    }
}

function addLog(message, type) {
    const timestamp = new Date().toLocaleTimeString();
    const logDiv = document.getElementById('verify_log');
    const icons = {
        'info': '🔍',
        'success': '✓',
        'error': '✗',
        'warning': '⚠'
    };
    const colors = {
        'info': 'text-muted',
        'success': 'text-success',
        'error': 'text-danger',
        'warning': 'text-warning'
    };
    
    const logEntry = '<div class="' + colors[type] + '">[' + timestamp + '] ' + icons[type] + ' ' + message + '</div>';
    logDiv.innerHTML += logEntry;
    logDiv.scrollTop = logDiv.scrollHeight;
}

function updateProgress(percent, file, status) {
    document.getElementById('overall_progress').style.width = percent + '%';
    document.getElementById('overall_progress').textContent = Math.round(percent) + '%';
    document.getElementById('overall_percent').textContent = Math.round(percent) + '%';
    document.getElementById('current_file').innerHTML = '<i class="bi bi-file-earmark-text"></i> ' + file;
    if (status) {
        document.getElementById('current_status').textContent = status;
    }
}

function updateStats() {
    document.getElementById('verified_files').textContent = verifiedCount;
    document.getElementById('missing_files').textContent = missingCount;
    document.getElementById('total_size').textContent = (totalSize / 1024 / 1024).toFixed(2) + ' MB';
}

async function verifyFile(filePath) {
    try {
        const response = await fetch('install_verify.php?file=' + encodeURIComponent(filePath));
        const result = await response.json();
        return result;
    } catch (error) {
        return {exists: false, size: 0, error: error.message};
    }
}

async function startVerification() {
    addLog('Starting file verification...', 'info');
    addLog('Total files to check: ' + totalFiles, 'info');
    
    for (let i = 0; i < filesToVerify.length; i++) {
        const file = filesToVerify[i];
        const progress = ((i + 1) / totalFiles) * 100;
        
        updateProgress(progress, file, (i + 1) + '/' + totalFiles);
        
        // Verify file exists
        const result = await verifyFile(file);
        
        if (result.exists) {
            verifiedCount++;
            totalSize += result.size;
            addLog('Found: ' + file + ' (' + (result.size / 1024).toFixed(1) + ' KB)', 'success');
        } else {
            missingCount++;
            missingFiles.push(file);
            addLog('MISSING: ' + file, 'error');
        }
        
        updateStats();
        
        // Small delay for visual effect
        await new Promise(resolve => setTimeout(resolve, 50));
    }
    
    addLog('Verification complete!', 'info');
    addLog('Files verified: ' + verifiedCount + '/' + totalFiles, verifiedCount === totalFiles ? 'success' : 'warning');
    
    if (missingCount === 0) {
        // All files present
        document.getElementById('verify_complete').style.display = 'block';
        addLog('All required files are present. Installation can proceed.', 'success');
        
        setTimeout(function() {
            window.location.href = 'install.php?step=3';
        }, 3000);
    } else if (missingCount <= 5) {
        // Some files missing (warning)
        document.getElementById('verify_warning').style.display = 'block';
        document.getElementById('warning_message').textContent = missingCount + ' files are missing. Installation may not work correctly.';
        
        let missingHtml = '<ul class="small mb-0">';
        missingFiles.forEach(function(file) {
            missingHtml += '<li><code>' + file + '</code></li>';
        });
        missingHtml += '</ul>';
        document.getElementById('missing_list').innerHTML = missingHtml;
        
        addLog('WARNING: ' + missingCount + ' files missing', 'warning');
    } else {
        // Too many files missing (error)
        document.getElementById('verify_error').style.display = 'block';
        document.getElementById('error_message').textContent = missingCount + ' critical files are missing. Please upload all files before installing.';
        addLog('ERROR: Too many files missing. Installation blocked.', 'error');
    }
}

function continueAnyway() {
    if (confirm('Warning: Some files are missing. The system may not work correctly.\n\nAre you sure you want to continue?')) {
        window.location.href = 'install.php?step=3';
    }
}

// Auto-start verification when page loads
window.addEventListener('load', function() {
    setTimeout(startVerification, 500);
});
</script>

<style>
.spinner-grow {
    animation: spinner-grow .75s linear infinite;
}
@keyframes spinner-grow {
    0% { transform: scale(0); }
    50% { transform: scale(1); }
}
</style>
