<?php
/**
 * RIU Integration for Installation Process
 * Include this file in install.php to enable progress reporting
 */

// Only load if in installation context
if (!defined('INSTALL_CONTEXT')) {
    define('INSTALL_CONTEXT', true);
}

// Load RIU helper only if database config exists
$riu_helper_loaded = false;
if (file_exists(__DIR__ . '/config/database.php') && file_exists(__DIR__ . '/includes/riu_helper.php')) {
    try {
        require_once __DIR__ . '/includes/riu_helper.php';
        $riu_helper_loaded = true;
    } catch (Exception $e) {
        // RIU helper not available yet - continue without it
        $riu_helper_loaded = false;
    }
}

// Installation progress tracking
function track_install_progress($step, $action = 'start') {
    global $riu_helper_loaded;

    // Only track if RIU helper is available
    if (!$riu_helper_loaded) {
        return;
    }

    static $steps = array(
        0 => 'System Requirements Check',
        1 => 'License Validation',
        2 => 'Installation Mode Selection',
        3 => 'File Operations',
        4 => 'Database Configuration',
        5 => 'Business Profile',
        6 => 'Admin Account Creation'
    );

    $total_steps = 7;
    $current_step_num = $step;

    if ($action === 'start') {
        // Report starting new step
        if (function_exists('monitor_system_state')) {
            monitor_system_state('installing', RIU_VERSION, array(
                'current_step' => $steps[$step] ?? 'Unknown Step',
                'step_number' => $current_step_num,
                'total_steps' => $total_steps
            ));
        }

        if (function_exists('report_installation_stage')) {
            report_installation_stage(
                $steps[$step] ?? 'Unknown Step',
                $current_step_num,
                $total_steps
            );
        }
    } elseif ($action === 'complete') {
        // Report step completion
        if (function_exists('log_system_event')) {
            log_system_event(
                'install',
                'completed',
                "Completed: " . ($steps[$step] ?? 'Unknown Step'),
                array(
                    'step' => $steps[$step] ?? 'Unknown Step',
                    'step_number' => $current_step_num,
                    'completed_at' => date('Y-m-d H:i:s')
                )
            );
        }
    } elseif ($action === 'error') {
        // Report error
        if (function_exists('log_system_event')) {
            log_system_event(
                'install',
                'failed',
                "Error at step: " . ($steps[$step] ?? 'Unknown Step'),
                array(
                    'step' => $steps[$step] ?? 'Unknown Step',
                    'step_number' => $current_step_num,
                    'error_at' => date('Y-m-d H:i:s')
                )
            );
        }
    }
}

// File download progress reporting (for remote install)
function report_download_progress($current, $total, $filename = '') {
    $percentage = $total > 0 ? round(($current / $total) * 100) : 0;

    log_system_event(
        'install',
        'running',
        "Downloading files: {$current}/{$total}",
        array(
            'action' => 'downloading',
            'current_file' => $filename,
            'files_downloaded' => $current,
            'total_files' => $total,
            'progress' => $percentage
        )
    );
}

// Database setup progress reporting
function report_database_progress($message, $tables_created = 0, $total_tables = 16) {
    $percentage = $total_tables > 0 ? round(($tables_created / $total_tables) * 100) : 0;

    log_system_event(
        'install',
        'running',
        $message,
        array(
            'action' => 'database_setup',
            'tables_created' => $tables_created,
            'total_tables' => $total_tables,
            'progress' => $percentage
        )
    );
}

// License verification progress
function report_license_check($status, $message = '') {
    log_system_event(
        'install',
        $status,
        "License verification: " . $message,
        array(
            'action' => 'license_check',
            'status' => $status
        )
    );
}

// Installation completion handler
function finalize_installation() {
    global $pdo;

    try {
        // Get database info
        $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'license_type' LIMIT 1");
        $stmt->execute();
        $license_type = $stmt->fetchColumn();

        $stmt = $pdo->prepare("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers");
        $stmt->execute();
        $worker_count = $stmt->fetchColumn();

        // Mark installation as complete
        complete_installation(RIU_VERSION);

        // Send final status
        log_system_event(
            'install',
            'completed',
            'Installation completed successfully',
            array(
                'completed_at' => date('Y-m-d H:i:s'),
                'license_type' => $license_type,
                'initial_workers' => $worker_count,
                'version' => RIU_VERSION
            )
        );

        // Store last heartbeat time
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value, setting_type)
                               VALUES ('last_heartbeat', NOW(), 'datetime')
                               ON DUPLICATE KEY UPDATE setting_value = NOW()");
        $stmt->execute();

    } catch (Exception $e) {
        // Silent fail - installation was successful, just reporting failed
    }
}

// Installation error handler
function handle_install_error($error_message, $step = null) {
    if ($step !== null) {
        track_install_progress($step, 'error');
    }

    fail_installation($error_message);
}
?>
