<?php
/**
 * Save License Information to Database
 * Called during installation Step 5
 */

function save_license_information($pdo, $session_data) {
    try {
        // Extract license data from session
        $license_key = $session_data['license_key'] ?? '';
        $domain = $session_data['domain'] ?? '';
        $license_details = $session_data['license_data']['license_details'] ?? array();

        if (empty($license_key) || empty($domain)) {
            return array('success' => false, 'error' => 'Missing license information');
        }

        $prefix = $session_data['db_prefix'] ?? 'scms_';

        // Store license key
        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('license_key', ?, 'string', 'license', 'System license key')");
        $stmt->execute(array($license_key));

        // Store license domain
        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('license_domain', ?, 'string', 'license', 'Licensed domain')");
        $stmt->execute(array($domain));

        // Store license type
        $license_type = $license_details['license_type'] ?? 'standard';
        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('license_type', ?, 'string', 'license', 'License type')");
        $stmt->execute(array($license_type));

        // Store worker limit
        $worker_limit = $license_details['worker_limit'] ?? 50;
        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('worker_limit', ?, 'integer', 'license', 'Maximum workers allowed')");
        $stmt->execute(array($worker_limit));

        // Store customer name
        $customer_name = $license_details['customer_name'] ?? '';
        if (!empty($customer_name)) {
            $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                                   VALUES ('license_customer', ?, 'string', 'license', 'License holder')");
            $stmt->execute(array($customer_name));
        }

        // Store expiry date if exists
        if (!empty($license_details['expiry_date'])) {
            $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                                   VALUES ('license_expiry', ?, 'date', 'license', 'License expiry date')");
            $stmt->execute(array($license_details['expiry_date']));
        }

        // Initialize health check timestamps
        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('last_health_check', NOW(), 'datetime', 'system', 'Last system health check')");
        $stmt->execute();

        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('system_health_status', 'healthy', 'string', 'system', 'Current system health status')");
        $stmt->execute();

        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('daily_check_status', 'pending', 'string', 'system', 'Daily check status')");
        $stmt->execute();

        // Store installation date
        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('installation_date', NOW(), 'datetime', 'system', 'System installation date')");
        $stmt->execute();

        // Initial resource usage
        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                               VALUES ('current_resource_usage', '0', 'integer', 'system', 'Current worker count')");
        $stmt->execute();

        // Log installation in audit
        $stmt = $pdo->prepare("INSERT INTO {$prefix}audit_logs (action, table_name, new_values, ip_address, user_agent, created_at)
                               VALUES ('system_installed', 'system_settings', ?, ?, ?, NOW())");
        $stmt->execute(array(
            'License: ' . $license_type . ', Domain: ' . $domain,
            $_SERVER['REMOTE_ADDR'] ?? 'installer',
            $_SERVER['HTTP_USER_AGENT'] ?? 'installer'
        ));

        // Store RIU configuration settings
        $riu_settings = array(
            array('riu_control_plane_url', 'https://supix.co.uk/sys/riu', 'string', 'riu', 'RIU Control Plane URL'),
            array('riu_product_id', 'scms_v3', 'string', 'riu', 'RIU Product Identifier'),
            array('riu_version', '3.0.0', 'string', 'riu', 'Current System Version'),
            array('riu_enabled', 'true', 'boolean', 'riu', 'Enable RIU Integration'),
            array('license_server_url', 'https://www.expremit.com/lic/api/validate.php', 'string', 'license', 'License Validation Server URL')
        );

        foreach ($riu_settings as $setting) {
            $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group, description)
                                   VALUES (?, ?, ?, ?, ?)");
            $stmt->execute($setting);
        }

        // Perform initial license validation with the server
        $license_server = 'https://www.expremit.com/lic/api/validate.php';
        $validation_data = array(
            'license_key' => $license_key,
            'domain' => $domain,
            'product' => 'scms',
            'worker_count' => 0
        );

        $ch = curl_init($license_server);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($validation_data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        $response = curl_exec($ch);
        curl_close($ch);

        if ($response !== false) {
            $result = json_decode($response, true);

            // Store last validation timestamp
            $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value, setting_type, setting_group)
                                   VALUES ('last_license_validation', NOW(), 'datetime', 'license')");
            $stmt->execute();

            // Log validation
            $stmt = $pdo->prepare("INSERT INTO {$prefix}audit_logs (action, table_name, new_values, ip_address, created_at)
                                   VALUES ('license_validated', 'system_settings', ?, 'installer', NOW())");
            $stmt->execute(array($result['valid'] ? 'valid' : 'invalid'));
        }

        return array('success' => true);

    } catch (Exception $e) {
        return array('success' => false, 'error' => $e->getMessage());
    }
}
?>
