<?php
/**
 * COMPLETE INSTALLATION FLOW - LOCAL AND REMOTE
 *
 * IMPORTANT: Both local and remote installations MUST communicate with RIU
 *
 * Why?
 * 1. License verification (required for both)
 * 2. Instance registration (track installations)
 * 3. Enable future updates (RIU needs to know what's installed)
 * 4. Installation telemetry (success/failure reporting)
 */

require_once __DIR__ . '/../includes/riu_config.php';
require_once __DIR__ . '/../includes/RIUClient.php';

/**
 * INSTALLATION FLOW - BOTH MODES
 *
 * LOCAL MODE:  Verify license → Register instance → Install from local files → Report completion
 * REMOTE MODE: Verify license → Register instance → Download from RIU → Report completion
 *
 * The only difference is WHERE the files come from!
 */

class InstallationManager
{
    private $riu;
    private $installMode; // 'local' or 'remote'
    private $version;

    public function __construct($licenseKey, $domain, $installMode = 'remote', $version = null)
    {
        $this->installMode = $installMode;
        $this->version = $version ?? RIU_DEFAULT_VERSION;

        // Initialize RIU Client (REQUIRED for both modes)
        $this->riu = new RIUClient(
            RIU_CONTROL_PLANE_URL,
            RIU_PRODUCT_ID,
            $licenseKey,
            $domain
        );
    }

    /**
     * Step 1: Verify License & Register Instance
     * ALWAYS REQUIRED - Both local and remote modes
     */
    public function verifyAndRegister()
    {
        echo "Verifying license with RIU Control Plane...\n";

        $result = $this->riu->verifyLicense();

        if (!$result['success']) {
            return [
                'success' => false,
                'error' => $result['error'] ?? 'License verification failed',
                'step' => 'verification'
            ];
        }

        // Store in session for future use
        $_SESSION['riu_verified'] = true;
        $_SESSION['riu_access_token'] = $this->riu->getAccessToken();
        $_SESSION['riu_instance_id'] = $this->riu->getInstanceId();
        $_SESSION['riu_version'] = $this->version;

        echo "✓ License verified\n";
        echo "✓ Instance registered: " . $this->riu->getInstanceId() . "\n";

        return [
            'success' => true,
            'instance_id' => $this->riu->getInstanceId(),
            'access_token' => $this->riu->getAccessToken()
        ];
    }

    /**
     * Step 2: Install Files
     * Different behavior based on mode
     */
    public function installFiles()
    {
        if ($this->installMode === 'local') {
            return $this->installFromLocal();
        } else {
            return $this->installFromRemote();
        }
    }

    /**
     * LOCAL MODE: Install from files already on server
     * BUT still reports to RIU what was installed
     */
    private function installFromLocal()
    {
        echo "Installing from local files...\n";

        // Report to RIU that installation started
        $this->reportOperationStart('install', 'local');

        // Files are already on server - just verify they exist
        $install_root = dirname(__DIR__);
        $required_files = ['index.php', 'config/database.php', 'includes/functions.php'];

        foreach ($required_files as $file) {
            if (!file_exists($install_root . '/' . $file)) {
                $error = "Required file missing: {$file}";
                $this->reportOperationFailure($error);

                return [
                    'success' => false,
                    'error' => $error,
                    'step' => 'file_verification'
                ];
            }
        }

        echo "✓ All required files present\n";

        // Optionally: Get manifest from RIU to verify local files
        $this->verifyLocalFilesWithManifest();

        return [
            'success' => true,
            'message' => 'Local files verified',
            'mode' => 'local'
        ];
    }

    /**
     * REMOTE MODE: Download files from RIU
     */
    private function installFromRemote()
    {
        echo "Downloading files from RIU...\n";

        // Report to RIU that installation started
        $this->reportOperationStart('install', 'remote');

        // Get manifest
        $manifest_result = $this->riu->getManifest($this->version);

        if (!$manifest_result['success']) {
            $this->reportOperationFailure('Failed to get manifest');

            return [
                'success' => false,
                'error' => 'Failed to get release manifest',
                'step' => 'manifest'
            ];
        }

        $files = $manifest_result['manifest']['files'] ?? [];
        $total_files = count($files);
        $downloaded = 0;
        $install_root = dirname(__DIR__);

        echo "Downloading {$total_files} files...\n";

        // Download each file
        foreach ($files as $file) {
            $file_path = $file['path'];
            $destination = $install_root . '/' . $file_path;

            echo "  Downloading: {$file_path}... ";

            $result = $this->riu->downloadFile($this->version, $file_path, $destination);

            if ($result['success']) {
                $downloaded++;
                echo "✓\n";

                // Set permissions
                if (isset($file['permissions'])) {
                    chmod($destination, octdec($file['permissions']));
                }

                // Report progress
                $this->reportProgress($downloaded, $total_files, $file_path);
            } else {
                echo "✗\n";
                $error = "Failed to download {$file_path}: " . ($result['error'] ?? 'Unknown error');
                $this->reportOperationFailure($error);

                return [
                    'success' => false,
                    'error' => $error,
                    'step' => 'download',
                    'file' => $file_path
                ];
            }
        }

        echo "✓ Downloaded {$downloaded}/{$total_files} files\n";

        return [
            'success' => true,
            'message' => "Downloaded {$downloaded} files successfully",
            'total' => $total_files,
            'mode' => 'remote'
        ];
    }

    /**
     * Step 3: Complete Installation
     * Reports to RIU that installation succeeded
     */
    public function completeInstallation()
    {
        echo "Completing installation...\n";

        // Report to RIU that installation completed successfully
        $this->reportOperationComplete();

        // Store version info for future updates
        $this->saveVersionInfo();

        echo "✓ Installation completed\n";
        echo "✓ Instance registered with RIU for updates\n";

        return [
            'success' => true,
            'message' => 'Installation completed successfully'
        ];
    }

    /**
     * Verify local files against RIU manifest (optional but recommended)
     */
    private function verifyLocalFilesWithManifest()
    {
        echo "Verifying local files with RIU manifest...\n";

        $manifest_result = $this->riu->getManifest($this->version);

        if (!$manifest_result['success']) {
            echo "  Warning: Could not verify files (manifest unavailable)\n";
            return false;
        }

        $files = $manifest_result['manifest']['files'] ?? [];
        $install_root = dirname(__DIR__);
        $verified = 0;
        $mismatches = [];

        foreach ($files as $file) {
            $file_path = $install_root . '/' . $file['path'];

            if (file_exists($file_path)) {
                $actual_checksum = hash_file('sha256', $file_path);

                if ($actual_checksum === $file['checksum']) {
                    $verified++;
                } else {
                    $mismatches[] = $file['path'];
                }
            }
        }

        if (count($mismatches) > 0) {
            echo "  Warning: " . count($mismatches) . " files have checksum mismatches\n";
        } else {
            echo "  ✓ All local files verified\n";
        }

        return count($mismatches) === 0;
    }

    /**
     * Save version info for future updates
     */
    private function saveVersionInfo()
    {
        $version_file = dirname(__DIR__) . '/version.json';

        $version_data = [
            'product' => RIU_PRODUCT_ID,
            'version' => $this->version,
            'installed_at' => date('Y-m-d H:i:s'),
            'instance_id' => $this->riu->getInstanceId(),
            'install_mode' => $this->installMode,
            'riu_url' => RIU_CONTROL_PLANE_URL
        ];

        file_put_contents($version_file, json_encode($version_data, JSON_PRETTY_PRINT));
    }

    /**
     * Report operation start to RIU
     */
    private function reportOperationStart($type, $source)
    {
        // TODO: Add telemetry endpoint to RIU API
        // POST /api/v1/telemetry/operations
        // { "type": "install", "source": "local/remote", "status": "running" }

        $_SESSION['operation_started'] = time();
    }

    /**
     * Report progress to RIU
     */
    private function reportProgress($current, $total, $currentFile)
    {
        // TODO: Add progress endpoint to RIU API
        // POST /api/v1/telemetry/events
        // { "stage": "downloading", "progress": 50, "file": "index.php" }
    }

    /**
     * Report operation completion to RIU
     */
    private function reportOperationComplete()
    {
        // TODO: Add telemetry endpoint to RIU API
        // POST /api/v1/telemetry/operations/{id}/complete
        // { "status": "completed", "duration": 120 }
    }

    /**
     * Report operation failure to RIU
     */
    private function reportOperationFailure($error)
    {
        // TODO: Add telemetry endpoint to RIU API
        // POST /api/v1/telemetry/operations/{id}/fail
        // { "status": "failed", "error": "..." }
    }
}

/**
 * USAGE EXAMPLE - COMPLETE FLOW
 */

// Initialize session
session_start();

// Get form data
$license_key = $_POST['license_key'] ?? '';
$domain = extract_root_domain($_SERVER['HTTP_HOST']);
$install_mode = $_POST['install_mode'] ?? 'remote'; // 'local' or 'remote'

// Create installation manager
$installer = new InstallationManager($license_key, $domain, $install_mode);

// Step 1: Verify license & register instance (ALWAYS REQUIRED)
$verify_result = $installer->verifyAndRegister();

if (!$verify_result['success']) {
    die("Error: " . $verify_result['error']);
}

// Step 2: Install files (local or remote)
$install_result = $installer->installFiles();

if (!$install_result['success']) {
    die("Error: " . $install_result['error']);
}

// Step 3: Database setup (your existing code)
// ... create database ...
// ... run migrations ...
// ... create admin user ...

// Step 4: Complete installation (report to RIU)
$complete_result = $installer->completeInstallation();

if ($complete_result['success']) {
    echo "Installation successful!\n";
    echo "Instance ID: " . $_SESSION['riu_instance_id'] . "\n";
    echo "Updates enabled via RIU\n";
}
