<?php
/**
 * SCMS v3.0 - Professional Installer
 * Complete 5-Step Installation with License Validation
 */

session_start();

// Graceful error handling for installer
if (isset($_GET['debug'])) {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(E_ALL);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    ini_set('error_log', __DIR__ . '/installer_errors.log');
}

// Disable mysqli exceptions for graceful error handling
mysqli_report(MYSQLI_REPORT_OFF);

$errors = array();
$success = '';
$current_step = isset($_GET['step']) ? (int)$_GET['step'] : 0;
$action = isset($_GET['action']) ? $_GET['action'] : '';

// Load RIU integration if license is set
if (isset($_SESSION['license_key']) && file_exists(__DIR__ . '/install_riu_integration.php')) {
    // Define RIU constants for installer
    if (!defined('RIU_CONTROL_PLANE_URL')) {
        define('RIU_CONTROL_PLANE_URL', 'https://supix.co.uk/sys/riu');
    }
    if (!defined('RIU_PRODUCT_ID')) {
        define('RIU_PRODUCT_ID', 'scms');
    }
    if (!defined('RIU_VERSION')) {
        define('RIU_VERSION', '3.0.0');
    }

    require_once __DIR__ . '/install_riu_integration.php';

    // Track current step if navigating normally
    if ($current_step > 0 && empty($action) && empty($_POST)) {
        track_install_progress($current_step, 'start');
    }
}

// Check if already installed - verify both config file AND database tables
$is_installed = false;
if (file_exists('config/database.php')) {
    try {
        require_once 'config/database.php';
        // Check if critical tables exist
        $check_tables = ['users', 'sponsored_workers', 'system_settings'];
        $tables_exist = 0;

        foreach ($check_tables as $table) {
            $result = $pdo->query("SHOW TABLES LIKE '" . DB_PREFIX . $table . "'");
            if ($result && $result->rowCount() > 0) {
                $tables_exist++;
            }
        }

        // If all 3 critical tables exist, consider it installed
        if ($tables_exist >= 3) {
            $is_installed = true;
        }
    } catch (Exception $e) {
        // Database connection failed or tables don't exist - treat as fresh install
        $is_installed = false;
    }
}

// If already installed and trying to access installation steps, redirect to Step 0 (reinstall page)
if ($is_installed && $current_step > 0 && $action !== 'reinstall') {
    header('Location: install.php?step=0');
    exit;
}

// Extract root domain function for license validation
function extract_root_domain($host) {
    // Remove port if present
    $host = explode(':', $host)[0];

    // Remove www. prefix
    $host = preg_replace('/^www\./', '', $host);

    // For localhost or IP addresses, return as-is
    if ($host === 'localhost' || filter_var($host, FILTER_VALIDATE_IP)) {
        return $host;
    }

    // Split domain into parts
    $parts = explode('.', $host);
    $count = count($parts);

    // If only one part (localhost) or IP, return as-is
    if ($count < 2) {
        return $host;
    }

    // Handle common two-part TLDs (co.uk, com.au, etc.)
    $two_part_tlds = array('co.uk', 'com.au', 'co.za', 'com.br', 'co.nz', 'ac.uk', 'gov.uk');

    if ($count >= 3) {
        $potential_tld = $parts[$count - 2] . '.' . $parts[$count - 1];
        if (in_array($potential_tld, $two_part_tlds)) {
            // Return domain.co.uk format (last 3 parts)
            return $parts[$count - 3] . '.' . $potential_tld;
        }
    }

    // Standard domain (last 2 parts: domain.com)
    return $parts[$count - 2] . '.' . $parts[$count - 1];
}

// Handle AJAX request to save install mode
if ($action === 'set_install_mode' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    $install_mode = trim($_POST['install_mode'] ?? '');

    if (in_array($install_mode, ['local', 'remote'])) {
        $_SESSION['install_mode'] = $install_mode;
        $_SESSION['install_max_step'] = 3; // Allow progression to Step 3

        // Track step 2 completion
        if (function_exists('track_install_progress')) {
            track_install_progress(2, 'complete');
        }

        echo json_encode(['success' => true, 'mode' => $install_mode]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Invalid install mode']);
    }
    exit;
}

// Handle file operations complete - allow progression to Step 4
if ($action === 'complete_files' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');

    if (isset($_SESSION['install_mode'])) {
        $_SESSION['install_max_step'] = 4; // Allow progression to Step 4

        // Track step 3 completion
        if (function_exists('track_install_progress')) {
            track_install_progress(3, 'complete');
        }

        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Invalid session']);
    }
    exit;
}

// Reinstall handler - REQUIRES LICENSE VALIDATION
if ($action === 'reinstall' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $confirm = trim($_POST['confirm_delete'] ?? '');
    $license_key = trim($_POST['license_key'] ?? '');
    // SECURITY: Force server-side ROOT domain detection - NEVER trust user input
    $domain = extract_root_domain($_SERVER['HTTP_HOST']);
    $validator_url = trim($_POST['validator_url'] ?? 'https://www.expremit.com/lic/api/validate.php');

    if ($confirm !== 'DELETE') {
        $errors[] = "Type DELETE to confirm reinstallation";
        $current_step = 0;
    } elseif (empty($license_key)) {
        $errors[] = "License key is required for reinstallation";
        $current_step = 0;
    } else {
        // VALIDATE LICENSE FIRST - Critical security check
        $validation = validate_license($license_key, $domain, $validator_url, 'installation');

        if (!$validation['success']) {
            $errors[] = "Invalid license: " . $validation['message'] . " - Reinstallation blocked for security.";
            $current_step = 0;
        } else {
            // License valid - proceed with reinstall

            // Register with RIU for reinstallation
            if (file_exists(__DIR__ . '/includes/RIUClient.php')) {
                try {
                    require_once __DIR__ . '/includes/RIUClient.php';
                    $riu_client = new RIUClient(
                        'https://supix.co.uk/sys/riu',
                        'scms',
                        $license_key,
                        $domain
                    );

                    $riu_result = $riu_client->verifyLicense();

                    if ($riu_result['success']) {
                        // Store RIU tokens for reinstall
                        $riu_access_token = $riu_result['data']['access_token'] ?? null;
                        $riu_instance_id = $riu_result['data']['instance_id'] ?? null;

                        // Report reinstallation event
                        if (!empty($riu_access_token) && !empty($riu_instance_id)) {
                            try {
                                $riu_client->reportStage('reinstalling', '3.0.0', [
                                    'step' => 'Reinstall Initiated',
                                    'step_number' => 0,
                                    'total_steps' => 7,
                                    'reinstall' => true
                                ]);
                            } catch (Exception $e) {
                                error_log('RIU reportStage failed during reinstall: ' . $e->getMessage());
                            }
                        }
                    }
                } catch (Exception $e) {
                    error_log('RIU exception during reinstall: ' . $e->getMessage());
                }
            }

            if (file_exists('config/database.php')) {
                require_once 'config/database.php';

                $tables = array('users', 'sponsored_workers', 'attendance_records', 'holiday_requests',
                    'holiday_entitlements', 'documents', 'work_locations', 'cos_records',
                    'home_office_reports', 'rtw_checks', 'audit_logs', 'system_settings',
                    'system_lists', 'modules_registry', 'schema_migrations', 'email_queue');

                // Use PDO for reinstall
                try {
                    $pdo->exec("SET FOREIGN_KEY_CHECKS = 0");
                    foreach ($tables as $table) {
                        $pdo->exec("DROP TABLE IF EXISTS `" . DB_PREFIX . $table . "`");
                    }
                    $pdo->exec("SET FOREIGN_KEY_CHECKS = 1");
                } catch (Exception $e) {
                    $errors[] = "Database cleanup failed: " . $e->getMessage();
                }

                unlink('config/database.php');
                if (file_exists('config/.sys')) {
                    unlink('config/.sys');
                }

                // Store validated license for reinstall
                $reinstall_data = [
                    'license_key' => $license_key,
                    'license_data' => $validation['data'],
                    'validator_url' => $validator_url,
                    'riu_access_token' => $riu_access_token ?? null,
                    'riu_instance_id' => $riu_instance_id ?? null,
                    'riu_registered' => isset($riu_result) && $riu_result['success']
                ];

                session_destroy();
                session_start();

                // Restore license info and RIU tokens for reinstall - start fresh install from Step 0
                $_SESSION['reinstall_license_key'] = $reinstall_data['license_key'];
                $_SESSION['reinstall_license_data'] = $reinstall_data['license_data'];
                $_SESSION['reinstall_validator_url'] = $reinstall_data['validator_url'];
                $_SESSION['reinstall_riu_access_token'] = $reinstall_data['riu_access_token'];
                $_SESSION['reinstall_riu_instance_id'] = $reinstall_data['riu_instance_id'];
                $_SESSION['reinstall_riu_registered'] = $reinstall_data['riu_registered'];

                // Redirect to Step 0 to start fresh installation flow
                header('Location: install.php?step=0&reset=success');
                exit;
            }
        }
    }
}

// Step validation
if ($current_step > 0) {
    $max_step = isset($_SESSION['install_max_step']) ? $_SESSION['install_max_step'] : 1;
    if ($current_step > $max_step) {
        header("Location: install.php?step=" . $max_step);
        exit;
    }
}

// CRITICAL: Re-validate license on every step after Step 1
if ($current_step > 1 && $current_step <= 5) {
    if (!isset($_SESSION['license_key']) || !isset($_SESSION['domain'])) {
        // License data missing - force back to Step 1
        error_log('Session lost - License key: ' . (isset($_SESSION['license_key']) ? 'SET' : 'NOT SET') . ', Domain: ' . (isset($_SESSION['domain']) ? 'SET' : 'NOT SET') . ', Session ID: ' . session_id());
        session_destroy();
        session_start();
        header('Location: install.php?step=1&error=license_expired');
        exit;
    }

    // Re-validate license only if not validated in last hour (reduce server load)
    $last_validation = $_SESSION['last_license_check'] ?? 0;
    if (time() - $last_validation > 3600) {
        $validator_url = $_SESSION['validator_url'] ?? 'https://www.expremit.com/lic/api/validate.php';
        $revalidation = validate_license($_SESSION['license_key'], $_SESSION['domain'], $validator_url, 'installation');

        if (!$revalidation['success']) {
            // License no longer valid - abort installation
            $errors[] = "License validation failed: " . $revalidation['message'] . " - Installation cannot proceed.";
            session_destroy();
            session_start();
            header('Location: install.php?step=1&error=license_invalid');
            exit;
        }

        // Update license data with fresh validation
        $_SESSION['license_data'] = $revalidation['data'];
        $_SESSION['last_license_check'] = time();
    }
}

// Check if coming from reinstall with validated license
if ($current_step == 1 && isset($_SESSION['reinstall_license_key'])) {
    // Restore license from reinstall
    $_SESSION['license_key'] = $_SESSION['reinstall_license_key'];
    $_SESSION['license_data'] = $_SESSION['reinstall_license_data'];
    $_SESSION['validator_url'] = $_SESSION['reinstall_validator_url'];
    $_SESSION['domain'] = extract_root_domain($_SERVER['HTTP_HOST']);
    $_SESSION['install_max_step'] = 2;

    // Restore RIU tokens from reinstall
    if (isset($_SESSION['reinstall_riu_access_token'])) {
        $_SESSION['riu_access_token'] = $_SESSION['reinstall_riu_access_token'];
    }
    if (isset($_SESSION['reinstall_riu_instance_id'])) {
        $_SESSION['riu_instance_id'] = $_SESSION['reinstall_riu_instance_id'];
    }
    if (isset($_SESSION['reinstall_riu_registered'])) {
        $_SESSION['riu_registered'] = $_SESSION['reinstall_riu_registered'];
    }

    // Clear reinstall flags
    unset($_SESSION['reinstall_license_key']);
    unset($_SESSION['reinstall_license_data']);
    unset($_SESSION['reinstall_validator_url']);
    unset($_SESSION['reinstall_riu_access_token']);
    unset($_SESSION['reinstall_riu_instance_id']);
    unset($_SESSION['reinstall_riu_registered']);

    // Auto-proceed to Step 2
    header('Location: install.php?step=2');
    exit;
}

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $current_step > 0) {

    // STEP 1: License Validation
    if ($current_step == 1) {
        $license_key = trim($_POST['license_key'] ?? '');
        // SECURITY: Force server-side ROOT domain detection - NEVER trust user input
        $domain = extract_root_domain($_SERVER['HTTP_HOST']);
        $validator_url = trim($_POST['validator_url'] ?? 'https://www.expremit.com/lic/api/validate.php');

        if (empty($license_key)) {
            $errors[] = "License key is required";
        } else {
            // Validate license with server
            $validation = validate_license($license_key, $domain, $validator_url, 'installation');

            if ($validation['success']) {
                $_SESSION['license_key'] = $license_key;
                $_SESSION['domain'] = $domain;
                $_SESSION['validator_url'] = $validator_url;
                $_SESSION['license_data'] = $validation['data'];
                $_SESSION['last_license_check'] = time();
                $_SESSION['install_max_step'] = 2;

                // REGISTER WITH RIU CONTROL PLANE
                if (file_exists(__DIR__ . '/includes/RIUClient.php')) {
                    try {
                        if (file_exists(__DIR__ . '/includes/riu_config.php')) {
                            require_once __DIR__ . '/includes/riu_config.php';
                        }
                        require_once __DIR__ . '/includes/RIUClient.php';
                        $riu_client = new RIUClient(
                            RIU_CONTROL_PLANE_URL, // Use constant defined above
                            'scms', // Product ID must match RIU release
                            $license_key,
                            $domain
                        );

                        // Verify and register with RIU
                        $riu_result = $riu_client->verifyLicense();

                        if ($riu_result['success']) {
                            // Save RIU tokens for later use
                            $_SESSION['riu_access_token'] = $riu_result['data']['access_token'] ?? null;
                            $_SESSION['riu_instance_id'] = $riu_result['data']['instance_id'] ?? null;
                            $_SESSION['riu_registered'] = true;

                            // Report installation started (only if tokens are set)
                            if (!empty($_SESSION['riu_access_token']) && !empty($_SESSION['riu_instance_id'])) {
                                try {
                                    $riu_client->reportStage('installing', '3.0.0', [
                                        'step' => 'License Validation',
                                        'step_number' => 1,
                                        'total_steps' => 7
                                    ]);
                                } catch (Exception $e) {
                                    error_log('RIU reportStage failed: ' . $e->getMessage());
                                }
                            }
                        } else {
                            // Log RIU verification failure
                            error_log('RIU verification failed: ' . ($riu_result['message'] ?? 'Unknown error'));
                            $_SESSION['riu_error'] = $riu_result['message'] ?? 'RIU verification failed';
                        }
                    } catch (Exception $e) {
                        // RIU registration failed, but don't block installation
                        error_log('RIU exception: ' . $e->getMessage());
                        $_SESSION['riu_error'] = 'Exception: ' . $e->getMessage();
                    }
                }

                // Track step 1 completion
                if (function_exists('track_install_progress')) {
                    track_install_progress(1, 'complete');
                }

                // Proceed to installation method selection (Step 2)
                header('Location: install.php?step=2');
                exit;
            } else {
                $errors[] = $validation['message'];
            }
        }
    }

    // STEP 3: File Operations (handled by JavaScript, no POST processing needed)

    // STEP 4: Database Configuration
    if ($current_step == 4) {
        $db_host = trim($_POST['db_host']);
        $db_name = trim($_POST['db_name']);
        $db_user = trim($_POST['db_user']);
        $db_pass = $_POST['db_pass'];
        $db_prefix = trim($_POST['db_prefix'] ?? 'scms_');

        if (empty($db_name)) $errors[] = "Database name is required";
        if (empty($db_user)) $errors[] = "Database username is required";

        if (empty($errors)) {
            // Try PDO connection for consistency
            try {
                $test_pdo = new PDO(
                    "mysql:host={$db_host};dbname={$db_name};charset=utf8mb4",
                    $db_user,
                    $db_pass,
                    [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
                );
                $test_pdo = null; // Close connection

                // Connection successful

                $_SESSION['db_host'] = $db_host;
                $_SESSION['db_name'] = $db_name;
                $_SESSION['db_user'] = $db_user;
                $_SESSION['db_pass'] = $db_pass;
                $_SESSION['db_prefix'] = $db_prefix;
                $_SESSION['install_max_step'] = 5;

                // Save database config
                save_database_config($db_host, $db_name, $db_user, $db_pass, $db_prefix);

                // Track step 4 completion
                if (function_exists('track_install_progress')) {
                    track_install_progress(4, 'complete');
                }

                header('Location: install.php?step=5');
                exit;

            } catch (PDOException $e) {
                // Graceful error handling
                $error_msg = $e->getMessage();

                // Make error message user-friendly
                if (strpos($error_msg, 'Access denied') !== false) {
                    $errors[] = "Database connection failed: Access denied. Please check your database username and password.";
                } elseif (strpos($error_msg, 'Unknown database') !== false) {
                    $errors[] = "Database connection failed: Database '{$db_name}' does not exist. Please create it first via cPanel.";
                } elseif (strpos($error_msg, 'Connection refused') !== false) {
                    $errors[] = "Database connection failed: Cannot connect to MySQL server. Please check if MySQL is running.";
                } else {
                    $errors[] = "Database connection failed: " . $error_msg;
                }
            }
        }
    }

    // STEP 5: Business Profile
    if ($current_step == 5) {
        // Handle AJAX request for Companies House lookup
        if (isset($_GET['ajax']) && $_GET['ajax'] === 'lookup' && isset($_POST['company_number'])) {
            header('Content-Type: application/json');
            $search_term = trim($_POST['company_number']); // Keep spaces for company names
            $result = fetch_companies_house_data($search_term);
            echo json_encode($result);
            exit;
        }

        $_SESSION['company_name'] = trim($_POST['company_name'] ?? '');
        $_SESSION['company_reg'] = trim($_POST['company_reg'] ?? '');
        $_SESSION['industry'] = trim($_POST['industry'] ?? '');
        $_SESSION['address'] = trim($_POST['address'] ?? '');
        $_SESSION['city'] = trim($_POST['city'] ?? '');
        $_SESSION['postcode'] = trim($_POST['postcode'] ?? '');
        $_SESSION['phone'] = trim($_POST['phone'] ?? '');
        $_SESSION['email'] = trim($_POST['email'] ?? '');

        // Handle logo upload or URL
        $logo_saved = false;
        $logo_path = '';

        // Check if file was uploaded
        if (isset($_FILES['logo_upload']) && $_FILES['logo_upload']['error'] === UPLOAD_ERR_OK) {
            $upload = $_FILES['logo_upload'];
            $allowed = array('image/png', 'image/jpeg', 'image/jpg', 'image/gif');

            if (in_array($upload['type'], $allowed) && $upload['size'] <= 2 * 1024 * 1024) {
                $ext = pathinfo($upload['name'], PATHINFO_EXTENSION);
                $filename = 'logo_' . time() . '.' . $ext;
                $destination = 'uploads/' . $filename;

                if (move_uploaded_file($upload['tmp_name'], $destination)) {
                    $logo_path = $filename;
                    $logo_saved = true;
                }
            }
        }
        // Check if URL was provided
        elseif (!empty(trim($_POST['logo_url'] ?? ''))) {
            $logo_url = trim($_POST['logo_url']);

            // Validate URL
            if (filter_var($logo_url, FILTER_VALIDATE_URL)) {
                // Try to download and save the logo
                $image_data = @file_get_contents($logo_url);

                if ($image_data !== false) {
                    $ext = pathinfo(parse_url($logo_url, PHP_URL_PATH), PATHINFO_EXTENSION);
                    if (empty($ext)) $ext = 'png';

                    $filename = 'logo_' . time() . '.' . $ext;
                    $destination = 'uploads/' . $filename;

                    if (file_put_contents($destination, $image_data)) {
                        $logo_path = $filename;
                        $logo_saved = true;
                    }
                }
            }
        }

        $_SESSION['company_logo'] = $logo_path;

        if (empty($_SESSION['company_name'])) {
            $errors[] = "Company name is required";
        } else {
            $_SESSION['install_max_step'] = 6;

            // Track step 5 completion
            if (function_exists('track_install_progress')) {
                track_install_progress(5, 'complete');
            }

            header('Location: install.php?step=6');
            exit;
        }
    }

    // STEP 6: Administrator Account
    if ($current_step == 6) {
        $admin_username = trim($_POST['admin_username'] ?? '');
        $admin_email = trim($_POST['admin_email'] ?? '');
        $admin_password = $_POST['admin_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        $admin_pin = trim($_POST['admin_pin'] ?? '');

        if (empty($admin_username)) $errors[] = "Username is required";
        if (empty($admin_email)) $errors[] = "Email is required";
        if (empty($admin_password)) $errors[] = "Password is required";
        if (strlen($admin_password) < 8) $errors[] = "Password must be at least 8 characters";
        if ($admin_password !== $confirm_password) $errors[] = "Passwords do not match";
        if (strlen($admin_pin) !== 4 || !ctype_digit($admin_pin)) $errors[] = "PIN must be exactly 4 digits";

        if (empty($errors)) {
            $_SESSION['admin_username'] = $admin_username;
            $_SESSION['admin_email'] = $admin_email;
            $_SESSION['admin_password'] = password_hash($admin_password, PASSWORD_BCRYPT);
            $_SESSION['admin_pin'] = password_hash($admin_pin, PASSWORD_BCRYPT);
            $_SESSION['install_max_step'] = 7;

            // Track step 6 completion
            if (function_exists('track_install_progress')) {
                track_install_progress(6, 'complete');
            }

            header('Location: install.php?step=7');
            exit;
        }
    }

    // STEP 7: Run Installation
    if ($current_step == 7 && isset($_POST['start_installation'])) {
        // FINAL LICENSE CHECK before touching database
        $validator_url = $_SESSION['validator_url'] ?? 'https://www.expremit.com/lic/api/validate.php';
        $final_check = validate_license($_SESSION['license_key'], $_SESSION['domain'], $validator_url);

        if (!$final_check['success']) {
            $errors[] = "CRITICAL: License validation failed at installation. " . $final_check['message'];
            session_destroy();
            session_start();
            header('Location: install.php?step=1&error=license_final_check_failed');
            exit;
        }

        // License valid - proceed with installation
        require_once 'config/database.php';

        $install_result = run_installation($pdo);

        if ($install_result['success']) {
            $success = "Installation completed successfully!";
            $_SESSION['installation_complete'] = true;

            // Track step 7 completion and final installation
            if (function_exists('track_install_progress')) {
                track_install_progress(7, 'complete');
            }

            // Mark installation as complete in RIU
            if (function_exists('finalize_installation')) {
                finalize_installation();
            }

            // Clear installation session data
            session_destroy();

            // Redirect to login page
            header('Location: admin/login.php?installed=success');
            exit;
        } else {
            $errors = $install_result['errors'];

            // Track installation error
            if (function_exists('handle_install_error')) {
                handle_install_error('Installation failed at final step', 6);
            }
        }
    }
}

// License validation function
function validate_license($license_key, $domain, $validator_url, $validation_type = 'installation') {
    $post_data = array(
        'license_key' => $license_key,
        'domain' => $domain,
        'product' => 'scms',
        'validation_type' => $validation_type
    );

    // Try cURL first
    if (function_exists('curl_init')) {
        $ch = curl_init();
        curl_setopt_array($ch, array(
            CURLOPT_URL => $validator_url,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($post_data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_SSL_VERIFYPEER => true
        ));

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($http_code == 200 && $response) {
            $data = json_decode($response, true);

            if ($data && isset($data['valid']) && $data['valid']) {
                return array('success' => true, 'data' => $data);
            } else {
                $message = isset($data['message']) ? $data['message'] : 'License validation failed';
                return array('success' => false, 'message' => $message);
            }
        }
    }

    // Fallback to file_get_contents
    $url = $validator_url . '?' . http_build_query($post_data);
    $response = @file_get_contents($url);

    if ($response) {
        $data = json_decode($response, true);

        if ($data && isset($data['valid']) && $data['valid']) {
            return array('success' => true, 'data' => $data);
        } else {
            $message = isset($data['message']) ? $data['message'] : 'License validation failed';
            return array('success' => false, 'message' => $message);
        }
    }

    return array('success' => false, 'message' => 'Unable to reach license server. Please check validator URL or try again later.');
}

// Fetch Companies House data
function fetch_companies_house_data($search_term) {
    $search_term = trim($search_term);

    // Check if it's a company number (numeric) or company name
    if (preg_match('/^[0-9A-Z]{1,8}$/i', str_replace(' ', '', $search_term))) {
        // It's a company number - direct lookup
        $company_number = strtoupper(str_replace(' ', '', $search_term));
        // Pad to 8 digits
        $company_number = str_pad($company_number, 8, '0', STR_PAD_LEFT);
        return fetch_company_by_number($company_number);
    } else {
        // It's a company name - search first
        return search_company_by_name($search_term);
    }
}

// Direct company lookup by number
function fetch_company_by_number($company_number) {
    $api_url = "https://api.company-information.service.gov.uk/company/" . $company_number;
    $api_key = 'a362067a-50b4-4ef3-86e3-87702a38e1e1'; // Companies House API key

    if (function_exists('curl_init')) {
        $ch = curl_init();
        curl_setopt_array($ch, array(
            CURLOPT_URL => $api_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTPHEADER => array('Accept: application/json'),
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            CURLOPT_HTTPAUTH => CURLAUTH_BASIC,
            CURLOPT_USERPWD => $api_key . ':'
        ));

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);

        if ($http_code === 200 && $response) {
            $data = json_decode($response, true);

            if ($data) {
                // Extract relevant fields
                $address = $data['registered_office_address'] ?? array();
                $full_address = trim(implode(', ', array_filter([
                    $address['address_line_1'] ?? '',
                    $address['address_line_2'] ?? ''
                ])));

                return array(
                    'success' => true,
                    'data' => array(
                        'company_name' => $data['company_name'] ?? '',
                        'company_number' => $data['company_number'] ?? '',
                        'address' => $full_address,
                        'city' => $address['locality'] ?? '',
                        'postcode' => $address['postal_code'] ?? '',
                        'status' => ucfirst(str_replace('-', ' ', $data['company_status'] ?? '')),
                        'type' => ucfirst(str_replace('-', ' ', $data['type'] ?? '')),
                        'sic_codes' => isset($data['sic_codes']) ? implode(', ', $data['sic_codes']) : ''
                    )
                );
            }
        } elseif ($http_code === 404) {
            return array('success' => false, 'message' => 'Company number not found in Companies House register');
        } else {
            return array('success' => false, 'message' => 'API Error: ' . ($curl_error ?: "HTTP $http_code"));
        }
    }

    return array('success' => false, 'message' => 'cURL not available on this server');
}

// Search company by name
function search_company_by_name($company_name) {
    $api_url = "https://api.company-information.service.gov.uk/search/companies?q=" . urlencode($company_name) . "&items_per_page=5";
    $api_key = 'a362067a-50b4-4ef3-86e3-87702a38e1e1'; // Companies House API key

    if (function_exists('curl_init')) {
        $ch = curl_init();
        curl_setopt_array($ch, array(
            CURLOPT_URL => $api_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTPHEADER => array('Accept: application/json'),
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            CURLOPT_HTTPAUTH => CURLAUTH_BASIC,
            CURLOPT_USERPWD => $api_key . ':'
        ));

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);

        if ($http_code === 200 && $response) {
            $data = json_decode($response, true);

            if ($data && isset($data['items']) && count($data['items']) > 0) {
                // Return list of matching companies
                $results = array();
                foreach ($data['items'] as $item) {
                    $results[] = array(
                        'company_name' => $item['title'] ?? '',
                        'company_number' => $item['company_number'] ?? '',
                        'status' => ucfirst(str_replace('-', ' ', $item['company_status'] ?? '')),
                        'address' => $item['address_snippet'] ?? ''
                    );
                }

                // If only one result, fetch full details
                if (count($results) === 1) {
                    return fetch_company_by_number($results[0]['company_number']);
                }

                return array('success' => true, 'multiple' => true, 'results' => $results);
            } else {
                return array('success' => false, 'message' => 'No companies found matching "' . htmlspecialchars($company_name) . '"');
            }
        } else {
            return array('success' => false, 'message' => 'Search API Error: ' . ($curl_error ?: "HTTP $http_code"));
        }
    }

    return array('success' => false, 'message' => 'cURL not available');
}

// Save database config
function save_database_config($host, $name, $user, $pass, $prefix) {
    $config = "<?php\n";
    $config .= "/**\n * SCMS v3.0 - Database Configuration\n * Auto-generated by installer\n */\n\n";
    $config .= "define('DB_HOST', '" . addslashes($host) . "');\n";
    $config .= "define('DB_NAME', '" . addslashes($name) . "');\n";
    $config .= "define('DB_USER', '" . addslashes($user) . "');\n";
    $config .= "define('DB_PASS', '" . addslashes($pass) . "');\n";
    $config .= "define('DB_PREFIX', '" . addslashes($prefix) . "');\n";
    $config .= "define('BASE_URL', '/sms');\n\n";
    $config .= "try {\n";
    $config .= "    \$pdo = new PDO(\n";
    $config .= "        'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4',\n";
    $config .= "        DB_USER,\n";
    $config .= "        DB_PASS,\n";
    $config .= "        [\n";
    $config .= "            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,\n";
    $config .= "            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,\n";
    $config .= "            PDO::ATTR_EMULATE_PREPARES => false\n";
    $config .= "        ]\n";
    $config .= "    );\n";
    $config .= "} catch (PDOException \$e) {\n";
    $config .= "    die('Database connection failed: ' . \$e->getMessage());\n";
    $config .= "}\n\n";
    $config .= "date_default_timezone_set('Europe/London');\n";
    $config .= "?>";

    file_put_contents('config/database.php', $config);
}

// Run installation
function run_installation($pdo) {
    $errors = array();
    $prefix = DB_PREFIX;

    // Create tables
    $tables = array(
        "CREATE TABLE IF NOT EXISTS `{$prefix}users` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `username` VARCHAR(100) UNIQUE NOT NULL,
            `email` VARCHAR(255) UNIQUE NOT NULL,
            `password_hash` VARCHAR(255) NOT NULL,
            `pin_hash` VARCHAR(255) NOT NULL,
            `permission_level` INT DEFAULT 100,
            `failed_attempts` INT DEFAULT 0,
            `status` VARCHAR(20) DEFAULT 'active',
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}sponsored_workers` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `employee_number` VARCHAR(50) UNIQUE NOT NULL,
            `full_name` VARCHAR(255) NOT NULL,
            `date_of_birth` DATE,
            `nationality` VARCHAR(100),
            `ni_number` VARCHAR(20),
            `department` VARCHAR(100),
            `job_title` VARCHAR(150),
            `manager_id` INT,
            `start_date` DATE,
            `contract_type` VARCHAR(50),
            `visa_type` VARCHAR(100),
            `visa_expiry` DATE,
            `cos_number` VARCHAR(100),
            `address` TEXT,
            `phone` VARCHAR(20),
            `email` VARCHAR(255),
            `emergency_contact` VARCHAR(255),
            `emergency_phone` VARCHAR(20),
            `photo` VARCHAR(255),
            `status` VARCHAR(20) DEFAULT 'active',
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX `idx_visa_expiry` (`visa_expiry`),
            INDEX `idx_department` (`department`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}attendance_records` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `worker_id` INT NOT NULL,
            `date` DATE NOT NULL,
            `status` VARCHAR(10) NOT NULL,
            `hours` DECIMAL(4,2) DEFAULT 0,
            `notes` TEXT,
            `created_by` INT,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY `unique_attendance` (`worker_id`, `date`),
            FOREIGN KEY (`worker_id`) REFERENCES `{$prefix}sponsored_workers`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}holiday_requests` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `worker_id` INT NOT NULL,
            `start_date` DATE NOT NULL,
            `end_date` DATE NOT NULL,
            `days` INT NOT NULL,
            `type` VARCHAR(50) NOT NULL,
            `is_paid` BOOLEAN DEFAULT TRUE,
            `reason` TEXT,
            `status` VARCHAR(20) DEFAULT 'pending',
            `reviewed_by` INT,
            `reviewed_at` TIMESTAMP NULL,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (`worker_id`) REFERENCES `{$prefix}sponsored_workers`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}holiday_entitlements` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `worker_id` INT NOT NULL,
            `year` INT NOT NULL,
            `total_days` INT DEFAULT 28,
            `used_days` INT DEFAULT 0,
            `remaining_days` INT DEFAULT 28,
            UNIQUE KEY `unique_entitlement` (`worker_id`, `year`),
            FOREIGN KEY (`worker_id`) REFERENCES `{$prefix}sponsored_workers`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}documents` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `worker_id` INT,
            `filename` VARCHAR(255) NOT NULL,
            `original_name` VARCHAR(255) NOT NULL,
            `category` VARCHAR(100) NOT NULL,
            `file_path` VARCHAR(500) NOT NULL,
            `file_size` INT,
            `mime_type` VARCHAR(100),
            `expiry_date` DATE,
            `is_verified` BOOLEAN DEFAULT FALSE,
            `verified_by` INT,
            `verified_at` TIMESTAMP NULL,
            `uploaded_by` INT,
            `uploaded_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (`worker_id`) REFERENCES `{$prefix}sponsored_workers`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}work_locations` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `name` VARCHAR(255) NOT NULL,
            `type` VARCHAR(50),
            `address` TEXT,
            `city` VARCHAR(100),
            `postcode` VARCHAR(20),
            `capacity` INT DEFAULT 0,
            `status` VARCHAR(20) DEFAULT 'active',
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}cos_records` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `cos_number` VARCHAR(100) UNIQUE NOT NULL,
            `issue_date` DATE NOT NULL,
            `assigned_to` INT,
            `status` VARCHAR(50) DEFAULT 'issued',
            `cost` DECIMAL(10,2) DEFAULT 0,
            `expiry_date` DATE,
            `notes` TEXT,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (`assigned_to`) REFERENCES `{$prefix}sponsored_workers`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}home_office_reports` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `worker_id` INT NOT NULL,
            `report_type` VARCHAR(100) NOT NULL,
            `absence_days` INT,
            `deadline` DATE,
            `status` VARCHAR(50) DEFAULT 'draft',
            `submitted_at` TIMESTAMP NULL,
            `acknowledged_at` TIMESTAMP NULL,
            `created_by` INT,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (`worker_id`) REFERENCES `{$prefix}sponsored_workers`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}rtw_checks` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `worker_id` INT NOT NULL,
            `check_date` DATE NOT NULL,
            `document_type` VARCHAR(100) NOT NULL,
            `document_number` VARCHAR(150),
            `expiry_date` DATE,
            `next_check_due` DATE,
            `verified_by` INT,
            `notes` TEXT,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (`worker_id`) REFERENCES `{$prefix}sponsored_workers`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}audit_logs` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `user_id` INT,
            `action` VARCHAR(255) NOT NULL,
            `details` TEXT,
            `ip_address` VARCHAR(50),
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX `idx_user` (`user_id`),
            INDEX `idx_created` (`created_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}system_settings` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `setting_key` VARCHAR(100) UNIQUE NOT NULL,
            `setting_value` TEXT,
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}system_lists` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `list_type` VARCHAR(100) NOT NULL,
            `list_value` VARCHAR(255) NOT NULL,
            `sort_order` INT DEFAULT 0,
            `status` VARCHAR(20) DEFAULT 'active',
            INDEX `idx_type` (`list_type`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}modules_registry` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `module_name` VARCHAR(100) UNIQUE NOT NULL,
            `module_version` VARCHAR(20),
            `status` VARCHAR(20) DEFAULT 'active',
            `installed_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}schema_migrations` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `version` VARCHAR(20) NOT NULL,
            `module` VARCHAR(100),
            `executed_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",

        "CREATE TABLE IF NOT EXISTS `{$prefix}email_queue` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `recipient` VARCHAR(255) NOT NULL,
            `subject` VARCHAR(500) NOT NULL,
            `body` TEXT NOT NULL,
            `status` VARCHAR(20) DEFAULT 'pending',
            `attempts` INT DEFAULT 0,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            `sent_at` TIMESTAMP NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4"
    );

    foreach ($tables as $sql) {
        try {
            $pdo->exec($sql);
        } catch (PDOException $e) {
            $errors[] = "Table creation failed: " . $e->getMessage();
        }
    }

    if (empty($errors)) {
        // Insert admin user
        $stmt = $pdo->prepare("INSERT INTO {$prefix}users (username, email, password_hash, pin_hash, permission_level) VALUES (?, ?, ?, ?, 100)");
        $stmt->execute([$_SESSION['admin_username'], $_SESSION['admin_email'], $_SESSION['admin_password'], $_SESSION['admin_pin']]);

        // Insert business settings
        $settings = array(
            array('company_name', $_SESSION['company_name']),
            array('company_reg', $_SESSION['company_reg']),
            array('industry', $_SESSION['industry']),
            array('address', $_SESSION['address']),
            array('city', $_SESSION['city']),
            array('postcode', $_SESSION['postcode']),
            array('phone', $_SESSION['phone']),
            array('email', $_SESSION['email']),
            array('company_logo', $_SESSION['company_logo'] ?? '')
        );

        $stmt = $pdo->prepare("INSERT INTO {$prefix}system_settings (setting_key, setting_value) VALUES (?, ?)");
        foreach ($settings as $setting) {
            $stmt->execute($setting);
        }

        // Insert license settings
        if (isset($_SESSION['license_key']) && isset($_SESSION['domain'])) {
            $license_settings = array(
                array('license_key', $_SESSION['license_key']),
                array('license_domain', $_SESSION['domain']),
                array('validator_url', $_SESSION['validator_url'] ?? 'https://www.expremit.com/lic/api/validate.php')
            );

            // Add license details if available
            if (isset($_SESSION['license_data']['license_details'])) {
                $details = $_SESSION['license_data']['license_details'];
                if (isset($details['license_type'])) {
                    $license_settings[] = array('license_type', $details['license_type']);
                }
                if (isset($details['worker_limit'])) {
                    $license_settings[] = array('worker_limit', $details['worker_limit']);
                }
                if (isset($details['customer_name'])) {
                    $license_settings[] = array('license_customer', $details['customer_name']);
                }
                if (isset($details['expiry_date'])) {
                    $license_settings[] = array('license_expiry', $details['expiry_date']);
                }
            }

            // Save license settings to database
            foreach ($license_settings as $setting) {
                $stmt->execute($setting);
            }

            // Save initial license validation timestamp
            $stmt->execute(array('last_license_validation', date('Y-m-d H:i:s')));
        }

        // Seed default data
        seed_default_data($pdo, $prefix);

        // Save license cache
        save_license_cache($_SESSION['license_data'], $_SESSION['license_key'], $_SESSION['validator_url']);
    }

    return array('success' => empty($errors), 'errors' => $errors);
}

// Seed default data
function seed_default_data($pdo, $prefix) {
    // Departments
    $departments = array('HR', 'IT', 'Finance', 'Operations', 'Care', 'Administration');
    $stmt = $pdo->prepare("INSERT INTO {$prefix}system_lists (list_type, list_value, sort_order) VALUES ('department', ?, ?)");
    foreach ($departments as $i => $dept) {
        $stmt->execute([$dept, $i]);
    }

    // Document types
    $doc_types = array('Passport', 'Visa', 'BRP', 'Right to Work', 'Contract', 'Qualification', 'DBS Check');
    $stmt = $pdo->prepare("INSERT INTO {$prefix}system_lists (list_type, list_value, sort_order) VALUES ('document_type', ?, ?)");
    foreach ($doc_types as $i => $type) {
        $stmt->execute([$type, $i]);
    }

    // Holiday types
    $holiday_types = array('Annual Leave', 'Sick Leave', 'Unpaid Leave', 'Maternity Leave', 'Paternity Leave', 'Compassionate Leave');
    $stmt = $pdo->prepare("INSERT INTO {$prefix}system_lists (list_type, list_value, sort_order) VALUES ('holiday_type', ?, ?)");
    foreach ($holiday_types as $i => $type) {
        $stmt->execute([$type, $i]);
    }
}

// Save license cache
function save_license_cache($data, $key, $validator) {
    $cache = array(
        'valid' => true,
        'timestamp' => time(),
        'license_key' => $key,
        'validator_url' => $validator,
        'license_type' => isset($data['license_details']['license_type']) ? $data['license_details']['license_type'] : '',
        'worker_limit' => isset($data['license_details']['worker_limit']) ? $data['license_details']['worker_limit'] : 0,
        'customer' => isset($data['license_details']['customer_name']) ? $data['license_details']['customer_name'] : ''
    );

    $encoded = base64_encode(json_encode($cache));
    file_put_contents('config/.sys', $encoded);
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SCMS v3.0 Installer <?php echo $current_step > 0 ? "- Step $current_step of 5" : ""; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        body { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; padding: 30px 0; }
        .installer-container { max-width: 900px; margin: 0 auto; }
        .installer-card { background: white; border-radius: 15px; box-shadow: 0 10px 40px rgba(0,0,0,0.3); overflow: hidden; }
        .installer-header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 40px; text-align: center; }
        .installer-body { padding: 50px; }
        .step-indicator { display: flex; justify-content: space-between; margin-bottom: 40px; position: relative; }
        .step-indicator:before { content: ''; position: absolute; top: 20px; left: 10%; right: 10%; height: 3px; background: #e0e0e0; z-index: 0; }
        .step { flex: 1; text-align: center; position: relative; z-index: 1; }
        .step-number { width: 50px; height: 50px; border-radius: 50%; background: #e0e0e0; color: #666; display: inline-flex; align-items: center; justify-content: center; font-weight: bold; font-size: 18px; margin-bottom: 8px; transition: all 0.3s; }
        .step.active .step-number { background: #667eea; color: white; box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4); }
        .step.completed .step-number { background: #28a745; color: white; }
        .step.completed .step-number:before { content: '\2713'; }
        .step-label { display: block; font-size: 13px; color: #666; font-weight: 500; }
        .requirement { padding: 15px; border-left: 4px solid #28a745; background: #f8f9fa; margin-bottom: 12px; border-radius: 4px; }
        .requirement.error { border-color: #dc3545; background: #fff5f5; }
        .requirement i { margin-right: 12px; font-size: 18px; }
        .alert-gradient { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; }
    </style>
</head>
<body>
    <div class="installer-container">
        <div class="installer-card">
            <div class="installer-header">
                <h1><i class="bi bi-shield-check"></i> SCMS v3.0</h1>
                <p class="mb-0">Sponsorship Compliance Management System</p>
                <small>Professional Installation Wizard</small>
            </div>

            <div class="installer-body">

                <?php if ($current_step > 0): ?>
                <!-- Step Indicator -->
                <div class="step-indicator">
                    <div class="step <?php echo $current_step >= 1 ? 'completed' : ''; ?> <?php echo $current_step == 1 ? 'active' : ''; ?>">
                        <div class="step-number"><?php echo $current_step > 1 ? '' : '1'; ?></div>
                        <span class="step-label">License</span>
                    </div>
                    <div class="step <?php echo $current_step >= 2 ? 'completed' : ''; ?> <?php echo $current_step == 2 ? 'active' : ''; ?>">
                        <div class="step-number"><?php echo $current_step > 2 ? '' : '2'; ?></div>
                        <span class="step-label">Database</span>
                    </div>
                    <div class="step <?php echo $current_step >= 3 ? 'completed' : ''; ?> <?php echo $current_step == 3 ? 'active' : ''; ?>">
                        <div class="step-number"><?php echo $current_step > 3 ? '' : '3'; ?></div>
                        <span class="step-label">Business</span>
                    </div>
                    <div class="step <?php echo $current_step >= 4 ? 'completed' : ''; ?> <?php echo $current_step == 4 ? 'active' : ''; ?>">
                        <div class="step-number"><?php echo $current_step > 4 ? '' : '4'; ?></div>
                        <span class="step-label">Admin</span>
                    </div>
                    <div class="step <?php echo $current_step >= 5 ? 'completed' : ''; ?> <?php echo $current_step == 5 ? 'active' : ''; ?>">
                        <div class="step-number"><?php echo $current_step > 5 ? '' : '5'; ?></div>
                        <span class="step-label">Install</span>
                    </div>
                </div>
                <?php endif; ?>

                <?php if (!empty($errors)): ?>
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle-fill"></i> <strong>Error:</strong>
                    <ul class="mb-0 mt-2">
                        <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>

                <?php if (!empty($success)): ?>
                <div class="alert alert-success">
                    <i class="bi bi-check-circle-fill"></i> <?php echo htmlspecialchars($success); ?>
                </div>
                <?php endif; ?>

                <?php if (isset($_GET['reset']) && $_GET['reset'] === 'success'): ?>
                <div class="alert alert-success">
                    <i class="bi bi-check-circle-fill"></i> Previous installation removed successfully!
                </div>
                <?php endif; ?>

                <!-- STEP CONTENT -->
                <?php
                // Handle installation steps
                if ($current_step == 3 && isset($_SESSION['install_mode'])) {
                    // Step 3 is file operations - use special file with progress bar
                    include_once "install_step3_files.php";
                } else if ($current_step == 1.5 || $current_step == 1.6 || $current_step == 1.7) {
                    // Handle decimal steps
                    $step_file = "install_step" . str_replace('.', '_', $current_step) . ".php";
                    include_once $step_file;
                } else {
                    // Normal step files
                    include_once "install_step{$current_step}.php";
                }
                ?>
            </div>
        </div>

        <div class="text-center mt-3">
            <small class="text-white">
                SCMS v3.0 &copy; 2025 | Support: <a href="mailto:abz@yours.com" class="text-white">Mr Awan</a>
            </small>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
