<?php
/**
 * Resource Manager
 * Handles resource allocation and capacity planning
 */

// Check if operation can proceed
function can_perform_operation($operation_type = 'default') {
    global $pdo;

    // Map operation types to resource checks
    $resource_operations = array(
        'add_worker',
        'add_user',
        'bulk_import',
        'create_location',
        'assign_cos'
    );

    if (in_array($operation_type, $resource_operations)) {
        require_once __DIR__ . '/system_health.php';

        // Check resource availability
        if ($operation_type === 'add_worker' || $operation_type === 'bulk_import') {
            $additional = ($operation_type === 'bulk_import') ? 5 : 1;
            $check = verify_resource_allocation($additional, $operation_type);

            if (!$check['allowed']) {
                return array(
                    'permitted' => false,
                    'message' => $check['reason'],
                    'upgrade_required' => true
                );
            }
        } elseif ($operation_type === 'add_user') {
            // Check with add_user validation type
            $check = verify_resource_allocation(0, 'add_user');

            if (!$check['allowed']) {
                return array(
                    'permitted' => false,
                    'message' => $check['reason'],
                    'upgrade_required' => true
                );
            }
        }
    }

    return array('permitted' => true, 'message' => 'Operation authorized');
}

// Pre-validate data operation
function validate_data_operation($entity_type, $action = 'create') {
    global $pdo;

    // Embedded health check for critical operations
    if ($entity_type === 'worker' && $action === 'create') {
        $permission = can_perform_operation('add_worker');

        if (!$permission['permitted']) {
            return array(
                'valid' => false,
                'error' => $permission['message'],
                'requires_upgrade' => $permission['upgrade_required'] ?? false
            );
        }
    }

    // Check data integrity
    try {
        // Verify database connection health
        $stmt = $pdo->query("SELECT 1");
        if (!$stmt) {
            return array('valid' => false, 'error' => 'Database connection error');
        }
    } catch (Exception $e) {
        return array('valid' => false, 'error' => 'System error');
    }

    return array('valid' => true);
}

// Get system capacity information
function get_system_capacity() {
    require_once __DIR__ . '/system_health.php';

    $current = get_resource_usage();
    $health = check_system_integrity('feature_check');

    if (!$health['status']) {
        return array(
            'current_usage' => $current,
            'capacity' => 'unknown',
            'status' => 'warning',
            'message' => 'System configuration needs attention'
        );
    }

    $details = $health['details'] ?? array();
    $limit = $details['worker_limit'] ?? 0;

    return array(
        'current_usage' => $current,
        'capacity' => $limit,
        'percentage' => $limit > 0 ? round(($current / $limit) * 100, 2) : 0,
        'status' => 'operational',
        'license_type' => $details['license_type'] ?? 'unknown'
    );
}

// Log resource operation
function log_resource_operation($operation_type, $entity_id = null, $details = '') {
    global $pdo;

    try {
        $user_id = $_SESSION['user_id'] ?? null;
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';

        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs
                               (user_id, action, table_name, record_id, new_values, ip_address, created_at)
                               VALUES (?, ?, 'resources', ?, ?, ?, NOW())");
        $stmt->execute(array($user_id, $operation_type, $entity_id, $details, $ip_address));
    } catch (Exception $e) {
        // Silent fail
    }
}

// Check if feature is available
function is_feature_available($feature_name) {
    require_once __DIR__ . '/system_health.php';

    $health = check_system_integrity('feature_check');

    if (!$health['status']) {
        // Limited features in degraded mode
        $basic_features = array('view_workers', 'view_attendance', 'view_reports');
        return in_array($feature_name, $basic_features);
    }

    return true;
}

// Initialize resource monitoring
function init_resource_monitor() {
    global $pdo;

    // Update resource usage statistics
    try {
        require_once __DIR__ . '/system_health.php';
        $current = get_resource_usage();

        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                               VALUES ('current_resource_usage', ?)
                               ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute(array($current, $current));

        // Trigger background health check
        init_health_monitor();

    } catch (Exception $e) {
        // Silent fail
    }
}
?>
