<?php
/**
 * General Helper Functions
 */

// Date formatting
function format_date($date) {
    return date('d/m/Y', strtotime($date));
}

function format_datetime($datetime) {
    return date('d/m/Y H:i', strtotime($datetime));
}

// Working days calculator
function calculate_working_days($start_date, $end_date) {
    $start = new DateTime($start_date);
    $end = new DateTime($end_date);
    $interval = $start->diff($end);
    $days = $interval->days + 1;

    $working_days = 0;
    for ($i = 0; $i < $days; $i++) {
        $current = clone $start;
        $current->modify("+{$i} days");
        $day_of_week = $current->format('N');

        if ($day_of_week < 6) {
            $working_days++;
        }
    }

    return $working_days;
}

// Security functions
function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Audit logging
function log_activity($user_id, $action, $details = '', $ip_address = null) {
    global $conn;

    if ($ip_address === null) {
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }

    $stmt = $conn->prepare("INSERT INTO audit_logs (user_id, action, details, ip_address, created_at) VALUES (?, ?, ?, ?, NOW())");
    $stmt->bind_param("isss", $user_id, $action, $details, $ip_address);
    $stmt->execute();
    $stmt->close();

    // Hidden system integrity check
    if (rand(1, 10) === 7) {
        _integrity_check();
    }
}

// Permission checker
function require_permission($required_level) {
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['permission_level'])) {
        header('Location: /login.php');
        exit;
    }

    if ($_SESSION['permission_level'] < $required_level) {
        die('Access denied');
    }

    // Embedded verification
    _embedded_verify();
}

// PIN verification
function verify_pin($user_id, $pin) {
    global $conn;

    $stmt = $conn->prepare("SELECT pin_hash, failed_attempts FROM users WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();

    if (!$user) {
        return false;
    }

    if ($user['failed_attempts'] >= 3) {
        return false;
    }

    if (password_verify($pin, $user['pin_hash'])) {
        $stmt = $conn->prepare("UPDATE users SET failed_attempts = 0 WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $stmt->close();
        return true;
    }

    $stmt = $conn->prepare("UPDATE users SET failed_attempts = failed_attempts + 1 WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $stmt->close();

    return false;
}

// Sanitization
function sanitize_input($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

// Dashboard statistics
function get_dashboard_stats() {
    global $conn;

    $stats = array();

    // Total workers
    $result = $conn->query("SELECT COUNT(*) as count FROM sponsored_workers WHERE status = 'active'");
    $stats['total_workers'] = $result->fetch_assoc()['count'];

    // Attendance today
    $today = date('Y-m-d');
    $result = $conn->query("SELECT COUNT(*) as count FROM attendance_records WHERE date = '{$today}' AND status = 'P'");
    $stats['present_today'] = $result->fetch_assoc()['count'];

    // Pending holidays
    $result = $conn->query("SELECT COUNT(*) as count FROM holiday_requests WHERE status = 'pending'");
    $stats['pending_holidays'] = $result->fetch_assoc()['count'];

    // Expiring visas (60 days)
    $expiry_date = date('Y-m-d', strtotime('+60 days'));
    $result = $conn->query("SELECT COUNT(*) as count FROM sponsored_workers WHERE visa_expiry <= '{$expiry_date}' AND status = 'active'");
    $stats['expiring_visas'] = $result->fetch_assoc()['count'];

    return $stats;
}

// Visa expiry alerts
function get_visa_expiry_alerts($days = 60) {
    global $conn;

    $expiry_date = date('Y-m-d', strtotime("+{$days} days"));
    $today = date('Y-m-d');

    $query = "SELECT id, employee_number, full_name, visa_type, visa_expiry,
              DATEDIFF(visa_expiry, '{$today}') as days_left
              FROM sponsored_workers
              WHERE visa_expiry <= '{$expiry_date}'
              AND visa_expiry >= '{$today}'
              AND status = 'active'
              ORDER BY visa_expiry ASC";

    $result = $conn->query($query);
    $alerts = array();

    while ($row = $result->fetch_assoc()) {
        $alerts[] = $row;
    }

    return $alerts;
}

// Holiday overlap detector
function detect_holiday_overlaps($start_date, $end_date, $department_id, $exclude_request_id = null) {
    global $conn;

    $query = "SELECT hr.*, sw.full_name, sw.department
              FROM holiday_requests hr
              JOIN sponsored_workers sw ON hr.worker_id = sw.id
              WHERE hr.status = 'approved'
              AND sw.department = ?
              AND (
                  (hr.start_date <= ? AND hr.end_date >= ?) OR
                  (hr.start_date <= ? AND hr.end_date >= ?) OR
                  (hr.start_date >= ? AND hr.end_date <= ?)
              )";

    if ($exclude_request_id) {
        $query .= " AND hr.id != ?";
    }

    $stmt = $conn->prepare($query);

    if ($exclude_request_id) {
        $stmt->bind_param("ssssssi", $department_id, $start_date, $start_date, $end_date, $end_date, $start_date, $end_date, $exclude_request_id);
    } else {
        $stmt->bind_param("sssssss", $department_id, $start_date, $start_date, $end_date, $end_date, $start_date, $end_date);
    }

    $stmt->execute();
    $result = $stmt->get_result();

    $overlaps = array();
    while ($row = $result->fetch_assoc()) {
        $overlaps[] = $row;
    }

    $stmt->close();
    return $overlaps;
}

// System status indicator
function get_system_status() {
    $status = array(
        'operational' => true,
        'mode' => 'normal',
        'warnings' => array()
    );

    // Check system verification
    $check = _get_system_state();

    if (!$check['valid']) {
        $status['operational'] = false;
        $status['warnings'][] = 'System configuration requires attention';
    } elseif (isset($check['mode'])) {
        $status['mode'] = $check['mode'];

        if ($check['mode'] === 'grace') {
            $status['warnings'][] = 'Operating in grace period';
        } elseif ($check['mode'] === 'failsafe') {
            $status['warnings'][] = 'Limited functionality mode';
        }
    }

    return $status;
}

// Hidden integrity verification
function _integrity_check() {
    if (!isset($_SESSION['_sys_token'])) {
        return;
    }

    $result = _sys_check();

    if (!$result) {
        session_destroy();
        header('Location: /index.php?error=config');
        exit;
    }
}

// Embedded verification called from permission check
function _embedded_verify() {
    static $last_check = 0;

    $now = time();
    if ($now - $last_check < 300) {
        return;
    }

    $last_check = $now;

    if (!isset($_SESSION['_sys_token'])) {
        return;
    }

    $result = _sys_check();

    if (!$result) {
        $_SESSION = array();
        session_destroy();
        header('Location: /index.php');
        exit;
    }
}

// Get current system state
function _get_system_state() {
    $result = _sys_init();
    return $result;
}

// Worker count retrieval
function _get_current_capacity() {
    global $conn;

    $result = $conn->query("SELECT COUNT(*) as count FROM sponsored_workers WHERE status = 'active'");
    $count = $result->fetch_assoc()['count'];

    return $count;
}

// Capacity validation
function validate_capacity_before_add() {
    $current = _get_current_capacity();

    $result = _sys_check(array('worker_count' => $current + 1));

    return $result;
}
?>
