<?php
/**
 * RIU Client - Remote Installation & Update System Client
 * Communicates with RIU Control Plane for license validation and file downloads
 */

class RIUClient
{
    private $controlPlaneUrl;
    private $productId;
    private $licenseKey;
    private $domain;
    private $accessToken;
    private $instanceId;

    /**
     * Initialize RIU Client
     *
     * @param string $controlPlaneUrl RIU Control Plane URL (e.g., https://supix.co.uk/sys/riu)
     * @param string $productId Product identifier (e.g., 'scms')
     * @param string $licenseKey License key
     * @param string $domain Installation domain
     */
    public function __construct($controlPlaneUrl, $productId, $licenseKey, $domain)
    {
        $this->controlPlaneUrl = rtrim($controlPlaneUrl, '/');
        $this->productId = $productId;
        $this->licenseKey = $licenseKey;
        $this->domain = $domain;
        $this->accessToken = null;
        $this->instanceId = null;
    }

    /**
     * Verify license with RIU Control Plane
     *
     * @return array ['success' => bool, 'message' => string, 'data' => array]
     */
    public function verifyLicense()
    {
        $url = $this->controlPlaneUrl . '/api/v1/auth/verify';

        $data = [
            'license_key' => $this->licenseKey,
            'domain' => $this->domain,
            'product' => $this->productId,
            'instance_id' => $this->generateInstanceId(),
        ];

        $response = $this->makeRequest($url, 'POST', $data);

        if ($response['success'] && isset($response['data']['access_token'])) {
            $this->accessToken = $response['data']['access_token'];
            $this->instanceId = $response['data']['instance_id'];

            // Save to session for later use
            $_SESSION['riu_access_token'] = $this->accessToken;
            $_SESSION['riu_instance_id'] = $this->instanceId;
        }

        return $response;
    }

    /**
     * Get release manifest
     *
     * @param string $version Version to download (e.g., '1.0.0')
     * @return array ['success' => bool, 'manifest' => array, 'signature' => string]
     */
    public function getManifest($version)
    {
        if (!$this->accessToken) {
            return ['success' => false, 'error' => 'Not authenticated. Call verifyLicense() first.'];
        }

        $url = $this->controlPlaneUrl . "/api/v1/releases/{$this->productId}/{$version}/manifest";

        return $this->makeRequest($url, 'GET', null, [
            'Authorization: Bearer ' . $this->accessToken
        ]);
    }

    /**
     * Download a single file
     *
     * @param string $version Version number
     * @param string $filePath File path from manifest
     * @param string $destinationPath Local destination path
     * @return array ['success' => bool, 'message' => string, 'checksum' => string]
     */
    public function downloadFile($version, $filePath, $destinationPath)
    {
        if (!$this->accessToken) {
            return ['success' => false, 'error' => 'Not authenticated. Call verifyLicense() first.'];
        }

        $url = $this->controlPlaneUrl . "/api/v1/releases/{$this->productId}/{$version}/download?" . http_build_query([
            'file' => $filePath
        ]);

        // Ensure destination directory exists
        $destDir = dirname($destinationPath);
        if (!is_dir($destDir)) {
            mkdir($destDir, 0755, true);
        }

        // Download file
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $this->accessToken
        ]);
        curl_setopt($ch, CURLOPT_HEADER, true);

        $response = curl_exec($ch);
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        $headers = substr($response, 0, $headerSize);
        $body = substr($response, $headerSize);

        curl_close($ch);

        if ($httpCode !== 200) {
            return [
                'success' => false,
                'error' => 'Download failed',
                'http_code' => $httpCode,
                'body' => $body
            ];
        }

        // Extract checksum from headers
        $checksum = null;
        if (preg_match('/X-File-Checksum:\s*([a-f0-9]+)/i', $headers, $matches)) {
            $checksum = $matches[1];
        }

        // Save file
        file_put_contents($destinationPath, $body);

        // Verify checksum
        $actualChecksum = hash_file('sha256', $destinationPath);
        if ($checksum && $actualChecksum !== $checksum) {
            unlink($destinationPath);
            return [
                'success' => false,
                'error' => 'Checksum verification failed',
                'expected' => $checksum,
                'actual' => $actualChecksum
            ];
        }

        return [
            'success' => true,
            'message' => 'File downloaded successfully',
            'checksum' => $actualChecksum,
            'path' => $destinationPath
        ];
    }

    /**
     * Download complete archive
     *
     * @param string $version Version number
     * @param string $destinationPath Local path to save archive
     * @return array ['success' => bool, 'message' => string, 'path' => string]
     */
    public function downloadArchive($version, $destinationPath)
    {
        if (!$this->accessToken) {
            return ['success' => false, 'error' => 'Not authenticated. Call verifyLicense() first.'];
        }

        $url = $this->controlPlaneUrl . "/api/v1/releases/{$this->productId}/{$version}/archive";

        // Ensure destination directory exists
        $destDir = dirname($destinationPath);
        if (!is_dir($destDir)) {
            mkdir($destDir, 0755, true);
        }

        // Download archive
        $ch = curl_init($url);
        $fp = fopen($destinationPath, 'w');

        curl_setopt($ch, CURLOPT_FILE, $fp);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $this->accessToken
        ]);

        curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        curl_close($ch);
        fclose($fp);

        if ($httpCode !== 200) {
            unlink($destinationPath);
            return [
                'success' => false,
                'error' => 'Archive download failed',
                'http_code' => $httpCode
            ];
        }

        return [
            'success' => true,
            'message' => 'Archive downloaded successfully',
            'path' => $destinationPath
        ];
    }

    /**
     * Extract downloaded archive
     *
     * @param string $archivePath Path to .tar.gz file
     * @param string $extractTo Extraction destination
     * @return array ['success' => bool, 'message' => string]
     */
    public function extractArchive($archivePath, $extractTo)
    {
        if (!file_exists($archivePath)) {
            return ['success' => false, 'error' => 'Archive file not found'];
        }

        // Ensure extraction directory exists
        if (!is_dir($extractTo)) {
            mkdir($extractTo, 0755, true);
        }

        // Try PharData (PHP's built-in tar handler)
        try {
            $phar = new PharData($archivePath);
            $phar->extractTo($extractTo, null, true);

            return [
                'success' => true,
                'message' => 'Archive extracted successfully',
                'path' => $extractTo
            ];
        } catch (Exception $e) {
            // Fallback to system tar command
            $command = sprintf(
                'tar -xzf %s -C %s',
                escapeshellarg($archivePath),
                escapeshellarg($extractTo)
            );

            exec($command, $output, $returnCode);

            if ($returnCode === 0) {
                return [
                    'success' => true,
                    'message' => 'Archive extracted successfully',
                    'path' => $extractTo
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to extract archive',
                'details' => $e->getMessage()
            ];
        }
    }

    /**
     * Generate unique instance ID based on server characteristics
     *
     * @return string Instance ID
     */
    private function generateInstanceId()
    {
        $factors = [
            $this->domain,
            $this->licenseKey,
            $_SERVER['SERVER_ADDR'] ?? 'unknown',
            $_SERVER['DOCUMENT_ROOT'] ?? 'unknown',
        ];

        return hash('sha256', implode('|', $factors));
    }

    /**
     * Make HTTP request to RIU Control Plane
     *
     * @param string $url URL
     * @param string $method HTTP method
     * @param array|null $data Request data
     * @param array $headers Additional headers
     * @return array Response
     */
    private function makeRequest($url, $method = 'GET', $data = null, $headers = [])
    {
        $ch = curl_init($url);

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
                $headers[] = 'Content-Type: application/json';
            }
        }

        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);

        if ($error) {
            return [
                'success' => false,
                'error' => 'Connection error: ' . $error
            ];
        }

        $decoded = json_decode($response, true);

        if ($httpCode >= 200 && $httpCode < 300) {
            return array_merge(['success' => true], $decoded ?: []);
        }

        return [
            'success' => false,
            'error' => $decoded['error'] ?? 'Request failed',
            'http_code' => $httpCode,
            'data' => $decoded
        ];
    }

    /**
     * Get access token (after successful verification)
     *
     * @return string|null
     */
    public function getAccessToken()
    {
        return $this->accessToken;
    }

    /**
     * Get instance ID
     *
     * @return string|null
     */
    public function getInstanceId()
    {
        return $this->instanceId;
    }

    /**
     * Report installation stage to RIU Control Plane
     *
     * @param string $stage Stage name: 'installing', 'active', 'maintenance', 'failed'
     * @param string $version Current version being installed/updated
     * @param array $metadata Additional metadata about the stage
     * @return array Response
     */
    public function reportStage($stage, $version, $metadata = [])
    {
        if (!$this->accessToken || !$this->instanceId) {
            return ['success' => false, 'error' => 'Not authenticated. Call verifyLicense() first.'];
        }

        $url = $this->controlPlaneUrl . '/api/v1/instances/heartbeat';

        $data = [
            'instance_id' => $this->instanceId,
            'status' => $stage,
            'current_version' => $version,
            'metadata' => $metadata,
        ];

        return $this->makeRequest($url, 'POST', $data, [
            'Authorization: Bearer ' . $this->accessToken
        ]);
    }

    /**
     * Report installation progress event
     *
     * @param string $operationType Type: 'install', 'update', 'rollback'
     * @param string $status Status: 'pending', 'running', 'completed', 'failed'
     * @param string $message Event message
     * @param array $metadata Additional event data
     * @return array Response
     */
    public function reportProgress($operationType, $status, $message, $metadata = [])
    {
        if (!$this->accessToken || !$this->instanceId) {
            return ['success' => false, 'error' => 'Not authenticated. Call verifyLicense() first.'];
        }

        $url = $this->controlPlaneUrl . '/api/v1/operations/report';

        $data = [
            'instance_id' => $this->instanceId,
            'type' => $operationType,
            'status' => $status,
            'message' => $message,
            'metadata' => $metadata,
        ];

        return $this->makeRequest($url, 'POST', $data, [
            'Authorization: Bearer ' . $this->accessToken
        ]);
    }

    /**
     * Check for pending operations (updates, maintenance, etc.)
     *
     * @return array ['success' => bool, 'operations' => array]
     */
    public function checkPendingOperations()
    {
        if (!$this->accessToken || !$this->instanceId) {
            return ['success' => false, 'error' => 'Not authenticated. Call verifyLicense() first.'];
        }

        $url = $this->controlPlaneUrl . '/api/v1/operations/pending';

        return $this->makeRequest($url, 'GET', null, [
            'Authorization: Bearer ' . $this->accessToken
        ]);
    }

    /**
     * Complete an operation (mark as completed or failed)
     *
     * @param int $operationId Operation ID from pending operations
     * @param string $status 'completed' or 'failed'
     * @param string|null $errorMessage Error message if failed
     * @return array Response
     */
    public function completeOperation($operationId, $status, $errorMessage = null)
    {
        if (!$this->accessToken) {
            return ['success' => false, 'error' => 'Not authenticated. Call verifyLicense() first.'];
        }

        $url = $this->controlPlaneUrl . '/api/v1/operations/' . $operationId . '/complete';

        $data = [
            'status' => $status,
            'error_message' => $errorMessage,
        ];

        return $this->makeRequest($url, 'POST', $data, [
            'Authorization: Bearer ' . $this->accessToken
        ]);
    }

    /**
     * Update instance heartbeat to show it's alive
     *
     * @param array $metadata Additional metadata (php_version, environment, etc.)
     * @return array Response
     */
    public function heartbeat($metadata = [])
    {
        if (!$this->accessToken || !$this->instanceId) {
            return ['success' => false, 'error' => 'Not authenticated. Call verifyLicense() first.'];
        }

        $url = $this->controlPlaneUrl . '/api/v1/instances/heartbeat';

        $data = [
            'instance_id' => $this->instanceId,
            'metadata' => array_merge([
                'php_version' => PHP_VERSION,
                'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'unknown',
                'memory_usage' => memory_get_usage(true),
            ], $metadata),
        ];

        return $this->makeRequest($url, 'POST', $data, [
            'Authorization: Bearer ' . $this->accessToken
        ]);
    }
}
