<?php
/**
 * Daily System Health Check
 * Run this via cron: 0 2 * * * /usr/bin/php /path/to/cron/daily_system_check.php
 */

// Load dependencies
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/system_health.php';
require_once __DIR__ . '/../includes/resource_manager.php';

echo "[" . date('Y-m-d H:i:s') . "] Starting daily system health check...\n";

try {
    // Perform comprehensive health check
    $health = check_system_integrity();

    if ($health['status']) {
        echo "[INFO] System health check PASSED\n";
        echo "[INFO] Status: " . $health['code'] . "\n";

        // Update system health status
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                               VALUES ('daily_check_status', 'healthy')
                               ON DUPLICATE KEY UPDATE setting_value = 'healthy'");
        $stmt->execute();

        // Get resource usage
        $usage = get_resource_usage();
        $capacity = get_system_capacity();

        echo "[INFO] Current resource usage: " . $usage . " / " . $capacity['capacity'] . "\n";
        echo "[INFO] Usage percentage: " . $capacity['percentage'] . "%\n";

        // Check if approaching limit
        if ($capacity['percentage'] >= 90) {
            echo "[WARNING] Resource usage is at " . $capacity['percentage'] . "% - approaching limit!\n";

            // Send notification to admins
            $stmt = $pdo->query("SELECT email FROM " . DB_PREFIX . "users WHERE role = 'admin' AND active = 1");
            $admins = $stmt->fetchAll(PDO::FETCH_COLUMN);

            foreach ($admins as $admin_email) {
                $subject = "SCMS Alert: Resource Capacity Warning";
                $message = "Your SCMS system is currently at " . $capacity['percentage'] . "% capacity.\n\n";
                $message .= "Current workers: " . $usage . "\n";
                $message .= "License limit: " . $capacity['capacity'] . "\n\n";
                $message .= "Please consider upgrading your license to avoid service interruption.\n";

                mail($admin_email, $subject, $message);
                echo "[INFO] Warning email sent to: " . $admin_email . "\n";
            }
        }

        // Log successful check
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs
                               (action, table_name, new_values, ip_address, created_at)
                               VALUES ('daily_health_check', 'system', 'passed', 'cron', NOW())");
        $stmt->execute();

    } else {
        echo "[ERROR] System health check FAILED\n";
        echo "[ERROR] Code: " . $health['code'] . "\n";

        // Update system health status
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                               VALUES ('daily_check_status', 'failed')
                               ON DUPLICATE KEY UPDATE setting_value = 'failed'");
        $stmt->execute();

        // Send critical alert to admins
        $stmt = $pdo->query("SELECT email FROM " . DB_PREFIX . "users WHERE role = 'admin' AND active = 1");
        $admins = $stmt->fetchAll(PDO::FETCH_COLUMN);

        foreach ($admins as $admin_email) {
            $subject = "SCMS Critical Alert: System Health Check Failed";
            $message = "Your SCMS system health check has failed.\n\n";
            $message .= "Error Code: " . $health['code'] . "\n";
            $message .= "Error Message: " . ($health['msg'] ?? 'Unknown error') . "\n\n";
            $message .= "Please log into your system and check the system status immediately.\n";

            mail($admin_email, $subject, $message);
            echo "[INFO] Alert email sent to: " . $admin_email . "\n";
        }

        // Log failed check
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs
                               (action, table_name, old_values, ip_address, created_at)
                               VALUES ('daily_health_check', 'system', ?, 'cron', NOW())");
        $stmt->execute(array($health['code']));
    }

    // Clean up old audit logs (keep last 90 days)
    echo "[INFO] Cleaning up old audit logs...\n";
    $stmt = $pdo->query("DELETE FROM " . DB_PREFIX . "audit_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY)");
    $deleted = $stmt->rowCount();
    echo "[INFO] Deleted " . $deleted . " old audit log entries\n";

    echo "[" . date('Y-m-d H:i:s') . "] Daily system health check completed successfully\n";

} catch (Exception $e) {
    echo "[ERROR] Exception occurred: " . $e->getMessage() . "\n";

    // Log exception
    try {
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs
                               (action, table_name, old_values, ip_address, created_at)
                               VALUES ('daily_health_check_error', 'system', ?, 'cron', NOW())");
        $stmt->execute(array($e->getMessage()));
    } catch (Exception $e2) {
        echo "[ERROR] Failed to log exception: " . $e2->getMessage() . "\n";
    }

    exit(1);
}

exit(0);
?>
