# SCMS Obfuscated License Validation System

## Overview

The SCMS license validation system is designed to be **distributed, obfuscated, and tamper-resistant** throughout the application. License checks are embedded in seemingly unrelated functions with non-obvious names to prevent easy bypassing.

---

## System Architecture

### Core Components

#### 1. **includes/system_health.php**
- **Purpose:** System integrity monitoring (disguised license validation)
- **Key Functions:**
  - `check_system_integrity()` - Validates license with remote server
  - `verify_resource_allocation($additional)` - Checks worker limits before operations
  - `get_resource_usage()` - Counts active workers
  - `run_background_health_check()` - Periodic validation (24-hour intervals)

#### 2. **includes/session_manager.php**
- **Purpose:** Session management with embedded license checks
- **Key Functions:**
  - `init_secure_session()` - Initializes session with system verification
  - `is_authenticated()` - Checks authentication with periodic re-verification (6 hours)
  - `verify_access($resource_type)` - Validates system status before granting access
  - `record_user_login($user_id)` - Triggers license validation on every login

#### 3. **includes/resource_manager.php**
- **Purpose:** Resource allocation with embedded capacity checks
- **Key Functions:**
  - `can_perform_operation($operation_type)` - Pre-validates operations (add_worker, bulk_import)
  - `validate_data_operation($entity_type, $action)` - Validates data operations
  - `get_system_capacity()` - Returns usage and limit information
  - `is_feature_available($feature_name)` - Feature gating based on license status

---

## Validation Points Throughout System

### Installation (Step 5)
- **File:** `install/save_license_info.php`
- **Action:** Stores license key, domain, limits in `system_settings` table
- **Validation:** Initial validation with license server
- **Logged:** Installation event in `audit_logs`

### User Login (Every Login)
- **File:** `admin/login.php` → calls `record_user_login()`
- **Action:** Silent license validation with current worker count
- **Frequency:** Every login attempt
- **Logged:** Login events in `audit_logs`

### Session Initialization (Every Page Load)
- **File:** `admin/header_include.php` → calls `init_secure_session()`
- **Action:** Checks if system verification is needed
- **Frequency:** Every 6 hours
- **Result:** Sets `$_SESSION['system_warning']` if license invalid

### Adding Workers
- **File:** `admin/workers/add_worker.php` → calls `validate_data_operation('worker', 'create')`
- **Action:** Validates if adding worker exceeds license limit
- **Result:** Blocks operation with upgrade message if limit reached
- **Logged:** Worker creation attempts

### Dashboard Load
- **Widget:** `admin/widgets/system_status.php`
- **Action:** Displays system capacity and license type
- **Frequency:** Every dashboard view
- **Visible:** Shows usage percentage and warnings at 75%, 90%

### Background Cron Job (Daily)
- **File:** `cron/daily_system_check.php`
- **Schedule:** 2:00 AM daily (configurable)
- **Action:** Comprehensive health check, sends email alerts
- **Logged:** Daily check results in `audit_logs`
- **Alerts:** Email notifications to admins if license invalid or capacity >90%

---

## Obfuscation Strategy

### Non-Obvious Function Names
| Function | Appears To Do | Actually Does |
|----------|---------------|---------------|
| `check_system_integrity()` | System diagnostics | License validation |
| `verify_resource_allocation()` | Resource planning | Worker limit checking |
| `get_system_status()` | Health monitoring | License status check |
| `init_secure_session()` | Session security | Embedded license check |
| `can_perform_operation()` | Permission check | License validation |

### Distributed Checks
- Not centralized in one "license" file
- Spread across: session, resource, health, cron
- Makes it difficult to find and disable all checks
- Each check has plausible alternative purpose

### Silent Failures
- Most checks allow **offline mode** (returns true if server unreachable)
- Graceful degradation instead of hard blocks
- Warnings logged but system continues functioning
- Critical blocks only for new resource allocation

---

## Data Storage

### Database Tables

#### system_settings
```sql
license_key          - License key string
license_domain       - Licensed domain
license_type         - Type: unlimited, paid, trial
worker_limit         - Maximum workers allowed
license_customer     - Customer name
license_expiry       - Expiry date (if applicable)
last_health_check    - Last validation timestamp
system_health_status - Current status: healthy, warning, failed
daily_check_status   - Daily cron status
current_resource_usage - Current worker count
last_license_validation - Last remote validation
```

#### audit_logs
All validation attempts, failures, and warnings are logged with:
- Action type (e.g., `system_health_warning`, `daily_health_check`)
- Timestamp
- IP address
- Error codes

---

## API Communication

### License Server Endpoint
`https://www.expremit.com/lic/api/validate.php`

### Request Parameters
```php
array(
    'license_key' => 'SCMS-XXXX-XXXX-XXXX-XXXX',
    'domain' => 'supix.co.uk',
    'product' => 'scms',
    'worker_count' => 15  // Optional, for limit checking
)
```

### Response (Valid)
```json
{
    "valid": true,
    "message": "License is valid!",
    "license_details": {
        "customer_name": "Organization Name",
        "license_type": "unlimited",
        "worker_limit": 999999,
        "status": "active",
        "expiry_date": "2026-12-31"
    }
}
```

### Response (Invalid)
```json
{
    "valid": false,
    "message": "License key not found"
}
```

---

## Validation Frequency

| Trigger | Frequency | Action |
|---------|-----------|--------|
| **Installation** | Once | Initial validation + store license info |
| **Login** | Every login | Silent validation with current usage |
| **Session Init** | Every 6 hours | Re-verification check |
| **Add Worker** | Before create | Pre-validate with new count |
| **Dashboard** | Every view | Display capacity widget |
| **Cron Job** | Daily (2 AM) | Comprehensive check + email alerts |

---

## Offline Mode

If license server is unreachable:
- `check_system_integrity()` returns `status: true, code: 'OFFLINE_MODE'`
- System continues functioning normally
- No blocks or restrictions
- User sees "Offline mode" in system status
- Allows temporary operation during server downtime

---

## Tamper Resistance

### Multiple Validation Points
- Removing one check doesn't disable system
- Checks are embedded in critical functions
- Functions have legitimate non-license purposes

### Database-Stored Configuration
- License info stored in database, not config files
- Harder to modify without database access
- Changes logged in audit trail

### Audit Logging
- All validation attempts logged
- Failed checks generate alerts
- Suspicious activity visible to admins

### Session-Based Tracking
- System verification status cached in session
- Periodic re-verification (6 hours)
- Can't bypass by refreshing page

---

## Implementation Checklist

- [x] Core validation wrapper functions created
- [x] Session manager with embedded checks
- [x] Resource manager with capacity validation
- [x] Login integration
- [x] Worker add integration
- [x] Dashboard widget
- [x] Background cron job
- [x] Installation integration
- [ ] Upload to production server
- [ ] Configure cron job
- [ ] Test all validation points
- [ ] Monitor audit logs

---

## Testing Validation

### Test Invalid License
1. Modify `license_key` in `system_settings` table to invalid value
2. Attempt login → Should see warning
3. Attempt to add worker → Should see error message
4. Check audit logs for failures

### Test Worker Limit
1. Set `worker_limit` to 5 in `system_settings`
2. Create 5 active workers
3. Attempt to add 6th worker → Should be blocked
4. Dashboard should show 100% capacity warning

### Test Offline Mode
1. Disconnect server from internet
2. Attempt login → Should work (offline mode)
3. System status should show "Offline mode"
4. Functionality should continue normally

---

## Security Considerations

1. **Never expose license validation logic** in client-side JavaScript
2. **Always validate server-side** before operations
3. **Log all validation attempts** for audit trail
4. **Use HTTPS** for license server communication
5. **Implement rate limiting** on validation endpoint
6. **Encrypt license keys** in database (optional)

---

## Maintenance

### Updating License Server URL
Change in all files:
- `includes/system_health.php` (line ~25)
- `install/save_license_info.php` (line ~90)

### Adjusting Validation Frequency
- **Session re-verification:** `includes/session_manager.php` line 30 (21600 = 6 hours)
- **Background check interval:** `includes/system_health.php` line 115 (24 hours)

### Email Notifications
- **Recipients:** Admins with `role = 'admin'` in `users` table
- **Configuration:** `cron/daily_system_check.php` lines 65-80

---

## File Summary

```
includes/
  ├── system_health.php          - Core validation (disguised as health monitoring)
  ├── session_manager.php         - Session with embedded checks
  └── resource_manager.php        - Resource allocation with capacity checks

admin/
  ├── header_include.php          - Require authentication + init monitoring
  ├── login.php                   - Login with validation
  ├── workers/add_worker.php      - Add worker with pre-validation
  └── widgets/system_status.php   - Dashboard widget

install/
  └── save_license_info.php       - Store license during installation

cron/
  └── daily_system_check.php      - Daily background validation
```

---

**Document Version:** 1.0
**Last Updated:** October 9, 2025
**Status:** Implementation Complete - Ready for Deployment
